# iprep_scores_import
#
# A cronjob that fetches IPRep and ASNRep score files from S3. Currently this
# is used by the globalviewcount hosts.
#
# @param env [Array] environments to use as consul tags.
# @param canary [String] additional courier install envs
# @param asnrep_db_dir [String] directory in which to place the asnrep db file.
# @param asnrep_db_filename [String] filename for the asnrep db inside the
#   $asnrep_db_dir.
# @param iprep_db_dir [String] directory in which to place iprep db files.
# @param run_on_update [String] path to a script that is run when either the
#   iprep or asnrep db's timestamp has changed after an update.
# @param run_on_update_delay [Integer] number of seconds to wait before running
#   the $run_on_update script.

class iprep_scores_import (
  $env                 = $iprep_scores_import::params::env,
  $canary              = $iprep_scores_import::params::canary,
  $asnrep_db_dir       = $iprep_scores_import::params::asnrep_db_dir,
  $asnrep_db_filename  = $iprep_scores_import::params::asnrep_db_filename,
  $iprep_db_dir        = $iprep_scores_import::params::iprep_db_dir,
  $run_on_update       = $iprep_scores_import::params::run_on_update,
  $run_on_update_delay = $iprep_scores_import::params::run_on_update_delay,
) inherits iprep_scores_import::params {
  validate_string($env)

  # dummy service to configure courier and consul similar to globalviewcount
  twitch::service { 'iprep-scores-import':
    daemon_cmd            => '/bin/sleep 365d',
    predaemon             => [],
    service_repo          => 'abuse/iprep-scores-import',
    env                   => $env,
    canary                => $canary,
    manage_service        => true,
    require               => [
        Iprep_scores_import::Dir[$iprep_db_dir],
        Iprep_scores_import::Dir[$asnrep_db_dir],
        Twitch_sandstorm_agent::Template['iprep-scores-import-secrets'],
    ],
  }

  twitch_cron { 'iprep-scores-import':
    environment => [
        'MAILTO=viewbots@twitch.tv',
        "ASNREP_DB_PATH='${asnrep_db_dir}/${asnrep_db_filename}'",
        "IPREP_DB_PATH='${iprep_db_dir}'",
        "RUN_ON_REP_UPDATE='${run_on_update}'",
        "RUN_ON_REP_UPDATE_DELAY='${run_on_update_delay}'",
    ],
    user        => 'root',
    command     => '/opt/twitch/iprep-scores-import/current/iprep-scores-import.sh | logger -p local3.info -t iprep-scores-import',
    minute      => '*',
    require     => Twitch::Service['iprep-scores-import'],
  }

  iprep_scores_import::dir { $asnrep_db_dir:
    path => $asnrep_db_dir,
  }

  iprep_scores_import::dir { $iprep_db_dir:
    path => $iprep_db_dir,
  }

  file { '/etc/iprep-scores-import':
    ensure => 'directory',
    path   => '/etc/iprep-scores-import',
    owner  => 'root',
    group  => 'root',
    mode   => '0700',
  }

  twitch_sandstorm_agent::template { 'iprep-scores-import-secrets':
    destination => '/etc/iprep-scores-import/iprep-scores-import.conf',
    contents    => file("${module_name}/iprep-scores-import.conf"),
    owner       => 'root',
    group       => 'root',
    mode        => '0600',
    require     => File['/etc/iprep-scores-import'],
  }
}
