#!/bin/bash
# This script adds a module to the Puppetfile of all currently available
# Environment branches in control-repo. It expects to be run inside the
# control-repo/ dir, works best if ../puppet-monolith exists and is current as well
# Use: add_to_all_envs.sh module_name [-g git_uri] [-v module_version]
# This defaults to tracking the production branch of the module unless
# -v is used to specify a version tag.
# Note: this will not make changes on the origin. It creates a couple of 
# scripts to handle that that must be run by hand

default_branch='production'
default_tracking=':control_branch'
project='video-puppet'
project3='video-puppet-thirdparty'
githost='git@git.xarth.tv'
function usage() {
    echo "usage: $0 <module> [-g git_uri] [-v module_version]"
    exit 
}

function error() {
    echo $1
    exit 1
}

if [[ $# -lt 1 ]]; then
    usage
fi
module=$1
shift

uri="${githost}:${project}/${module}.git"

while getopts "g:v:3" OPT ; do
    case $OPT in
        g)
            uri=$OPTARG
            ;;
        v)
            ver=$OPTARG
            ;;
        3)
            uri="${githost}:${project3}/${module}.git"
            ;;
        *)
            usage
            ;;
    esac
done

if ! $(ls -1d .git > /dev/null 2>&1) || ! $(ls -1 ./Puppetfile >/dev/null 2>&1); then
    error "Must be in a clone of control-repo"
fi
git fetch --prune

curr_branch=`git rev-parse --abbrev-ref HEAD`
if [[ $curr_branch != 'production' ]] ; then
    error "Must be on production branch of control-repo!"
fi

if ! git ls-remote $uri >/dev/null 2>&1 ; then
    echo "Make sure ${module} is a valid module and its empty repo exists in ${uri}. Create and hit enter to continue"
    read
fi

#auto-version
##if [[ -z "$ver" ]] ; then
##    if [[ -f ../puppet-monolith/modules/$module/metadata.json ]] ; then
##        checkver=$(jq .version ../puppet-monolith/modules/$module/metadata.json|sed -e 's/"//g')
##        tagvers=$(git ls-remote --tag --ref $uri '*'$checkver'*' --format | sed -e 's/.*refs\/tags\///')
##    
##        echo "Metadata in puppet-monolith module shows version ${checkver}, similar tags in new repo found these versions:"
##        echo "$tagvers"
##        echo "Enter the tag version to use or leave blank to follow 'production' branch:"
##        while read x; do
##            [ -z "${x}" ] && break
##            if git ls-remote --tag --ref $uri "${x}"|grep "${x}" ; then
##                ver=$x
##                break
##            fi
##            echo 'Invalid tag, type it as it is listed please, try again.'
##        done
##    fi
##fi

# Get all branches EXCEPT production (we PR to that one) and HEAD (not really a branch)
branches=`git for-each-ref refs/remotes --format='%(refname:short)'|grep -v '^origin/HEAD'|grep -v '^origin/production$'`
if [ ! $? ] ; then
    error "Error: unable to fetch branches. Check that git remotes is set properly!"
fi

# echo "curr_branch:"$curr_branch
# Create our migration branch to PR production if it doesn't exist, make sure we always update it
copt='-q'
if ! echo $branches | grep -q 'module_migration_pr_branch' ; then
    branches="${branches}"$'\n'"module_migration_pr_branch"
fi
if ! git branch | grep -q 'module_migration_pr_branch' ; then
    copt='-qb'
fi

block="
mod '${module}',
      :git            => '${uri}',
"

if [ -n "${ver}" ] ; then
    block="${block}      :tag            => '${ver}'"
else
    block="${block}      :default_branch => '${default_branch}',
      :branch         => ${default_tracking}"
fi

for branch in ${branches[@]} ; do
    echo "Current Branch ${branch}"
    checkout_opt='-q'
    if [[ ${branch} == "module_migration_pr_branch" ]] ; then
        checkout_opt="${copt}"
    fi
    if [[ ${branch} != 'production' ]]; then
        c_branch="${branch#origin/}"
        git checkout $checkout_opt "${c_branch}" >/dev/null 2>&1 || error "Error checking out ${branch}. Check that it is ok."
        if ! merge=$(git merge 'production' 2>&1) ; then
            merge_warns="${merge_warns}, ${branch}"
            git checkout --ours Puppetfile && git add Puppetfile && git commit -m 'Merging with Production branch changes'
            if [ ! $? ] ; then
                merge_fails="${merge_fails}, ${branch}"
                error "Error merging ${branch} with production, check that its not in a strange state"
            fi
        fi
        if grep -q "mod '${module}'," ./Puppetfile; then
            echo "Module ${module} already in Puppetfile for env ${c_branch}"
        else
            echo "Updating Puppetfile for ${c_branch}"
            echo "${block}" >> ./Puppetfile
            git commit -am "Adding module $module to Puppetfile"
        fi
        git checkout production || error "Error switching back to production branch from ${c_branch}"
    fi
done
echo
# create cleanup branch for migration
if [ -d ../monolith/puppet-monolith/modules/$module/ ] ; then
    pushd ../monolith/puppet-monolith/modules/
    git checkout master
    git pull
    if git ls-remote --heads |grep 'module_migration_pr_branch' || git branch | grep 'module_migration_pr_branch'; then
        gopt='-q'
    else
        gopt='-qb'
    fi
    git checkout $gopt module_migration_pr_branch
    if ! git status | grep 'module_migration_pr_branch'; then
        error 'Failed to create or switch to cleanup branch in puppet-monolith'
    fi
    git rm -rf $module
    mkdir $module
    md_uri=$(echo $uri|sed -e 's/_/\_/g')
    cat >> $module/README.md <<REOF
<!-- START doctoc generated TOC please keep comment here to allow auto update -->
<!-- DON'T EDIT THIS SECTION, INSTEAD RE-RUN doctoc TO UPDATE -->
**Table of Contents**  *generated with [DocToc](https://github.com/thlorenz/doctoc)*

- [This Module has Moved!](#this-module-has-moved)

<!-- END doctoc generated TOC please keep comment here to allow auto update -->

# This Module has Moved!
Find it here instead: ${md_uri}
REOF
    git add $module
    git commit -m "cleaning up  migration of $module to repo+puppetfile"
    echo
    echo "Once the PR is merged in control-repo for this branch (and all commits are pushed to existing Environments)"
    echo "Run the following line to cleanup the module/s, removing it/them from puppet-monolith."
    echo "Be sure to PR and merge before continuing, and remove the branch after merging!"
    echo
    echo "git push -u origin module_migration_pr_branch"
    echo
    git checkout master
    popd
fi
if [[ -n $merge_warns ]] ; then
    echo "Double check we did the right thing with these, the merge hit an exception!"
    echo $merge_warn|sed -e 's/^, //'
fi
if [[ -n $merge_fails ]] ; then
    echo "Manually check these control-repo branches, merge to production failed!"
    echo $merge_fails|sed -e 's/^, //'
fi
