#!/bin/bash
# This script relies on git-filter-repo: https://github.com/newren/git-filter-repo
# This script takes a puppet module name (name of the actual module path under modules/)
# and turns it into a stand-alone repository with full git history/branches/etc
# It sets the proper remotes for the new repo, but does not push upstream or create the origin repo
# This script does not push any changes to origin and is generally safe to run anytime.

# This works in a timestamped dir it creates, since git is a time-sensitive thing.
# It can be run from anywhere, tho suggested to run it inside a dir just for it (ie: ~/repo_migrations).
# Once the repo is pushed to origin, and origin (github) shows content in the repo,
# it is safe to rm -rf the dir, as the next run will create it again anyway.

# When running this utility, it is necessary to let anyone working on the module being migrated
# know about the move, and when it is happening, so they can sync any branches prior
# to the move, or know to sync them into the new repo. Notification is required again before
# the add_to_all_envs script is used to enable the results.

# The location of the git-filter-repo script is currently its repo path 
# ( ~/git/git-filter-repo/git-filter-repo), change it if
# you install the utility somewhere else.
GIT_FILTER_REPO=$(realpath ~/git/git-filter-repo/git-filter-repo)
ts=$(date +%s)
mirror_dir='puppet-monolith.git'

function error() {
    echo $1
    exit 1
}

if [[ $1 == '-t' ]] ; then
    shift
    ts=$1
    shift
fi

modules=$@

dir="migration_dir_${ts}"

if [[ $# -lt 1 ]] ; then
    error "usage: $0 [-t timestamp] <module> [module2] [module3] ..."
fi

[[ ! -d $dir ]] && mkdir $dir

if [[ ! -d $mirror_dir ]] ; then
    echo "Cloning bare repo puppet-monolith from remote origin"
    git clone --bare git@git.xarth.tv:video-puppet/puppet-monolith.git || error "Error cloning bare repo!"
fi

finish="Please make sure the repo exists in github and run this if it looks good:"

pushd $dir

for module in $modules ; do
    git clone ../$mirror_dir || error "Error cloning monolith from local bare repo!"
    if [[ ! -d puppet-monolith/modules/$module ]] ; then
        echo "Error! No module named '$module' present in puppet-monolith! skipping"
        continue
    fi
    echo > "${module}_diff_branches"
    echo > "${module}_no_diff_branches"
    pushd puppet-monolith/
    $GIT_FILTER_REPO --prune-empty=always --prune-degenerate=always --subdirectory-filter modules/$module || error "Error filtering repo for ${module}"

    for i in `git branch|grep -v '^\* '|sed -e 's/  //'` ; do
        echo $i
        git checkout $i || (echo "Error checkout out branch $i!" && continue)
        d=`git diff master... --name-only --diff-filter=MARC`
        if [[ -n "$d" ]]; then 
            echo "$i" >> "../${module}_diff_branches"
        else
            echo "$i" >> "../${module}_no_diff_branches"
        fi
    done
    git checkout master;
    for d in `cat "../${module}_no_diff_branches"` ; do
        git branch -D $d
    done
    if [ -n $rename_master ] ; then
        echo "Moving master -> production"
        if git branch |grep 'production' >/dev/null 2>&1 ; then
            if git branch |grep 'production_pre_migration' >/dev/null 2>&1 ; then
                error "production_pre_migration branch exists for ${module}! Evaluate by hand."
            fi
            echo "*** production branch exists in ${module}, moving it aside to production_pre_migration"
            git branch -m production production_pre_migration
        fi
        git branch -m master production
    fi

    git remote add origin "git@git.xarth.tv:video-puppet/${module}.git"

    popd
    mv puppet-monolith $module
    finish="${finish}
pushd ${dir}/${module}; git push origin --all --force; popd"
done
echo "${finish}"
popd
