# NRDC Puppet Module

Provides a means to install and configure NRDC with service health checks.
This is a passive check agent generally used with Nagios/NRDP and derivatives.


## Usage

When following one of the examples below, be sure to change the example token
to the correct value used for NRDP.

Include the module in hiera, like this:
```yaml
---
classes:
 - nrdc

nrdc::receivers:
  'http://nagios.company.com/nrdp/':
    http_vars: 'token=234werd123-s1234-a1234a123a-lllkju&cmd=submitcheck'
```

Or include it in a module:
```ruby
include nrdc

nrdc::receiver { 'http://nagios.company.com/nrdp/':
  http_vars => 'token=234werd123-s1234-a1234a123a-lllkju&cmd=submitcheck',
}
```

Another way:
```ruby
class { 'nrdc':
  receivers => {
    'http://nagios.company.com/nrdp/' => {
      http_vars => 'token=234werd123-s1234-a1234a123a-lllkju&cmd=submitcheck',
    },
  },
}
```

In addition to including the module and defining receivers, service checks should
also be created. If you have a base set of checks, it's sometimes useful to define
them in hiera like this:

```yaml
nrdc::servicechecks:
  'load':
    service_name: 'load'
    command: 'check_load -w 1 -c 2'
    timeout: 60
    interval: 5
    retry_interval: 2
    retry: 3
    user: 'nobody'
  'disk':
    command: 'check_disk / -w 70% -c 90%'
```

The above example shows two standard service checks being created using hiera;
the first one duplicates the default options. The only required parameters are
the `service_name`, which can be a hash key, and the `command`.

Service checks can also be created in manifests, typically right next to the service
they check. Like this:

```ruby
nrdc::servicecheck { 'memory':
  command => 'check_mem -w 50% -c 70%',
}
```


## Notes

As of this version of the NRDC puppet module, NRDC has native support for
NRDP and Slack as receiver types. It has a somewhat extensible interface that
you may get to work with other receivers. That support is available in this
module if you're trying to feed service checks into something besides NRDP.


## Parameters

These are the parameters you can pass into the `nrdc` class
directly or via hiera overrides.

#### `ensure` (default: `present`)
Setting this to absent will remove all of NRDC.
#### `config_file` (default: `/etc/nrdc/nrdc.conf`)
The default location of the config file we write our data to.
#### `include_dir` (default: `/etc/nrdc/conf.d`)
The location extra config file are written to.
#### `service_path` (default: `/var/lib/nrdc/services.d`)
The path passed to NRDC as service_path.
NRDC will watch this folder for services to check.
#### `results_path` (default: `/var/lib/nrdc/checkresults`)
The path passed to NRDC as results_path.
NRDC will watch this folder for external service results.
#### `manage_service` (default: `true`)
If set to true, puppet will manage the nrdc service.
#### `package_version` (default: `latest`)
Allows pinning and staging different versions.
#### `allowed_users` (default: `['nobody']`)
Only users in this list are allowed to run checks. Checks that define a username
not in this list will not be run.
#### `default_user` (default: `nobody`)
The user that runs service checks that do not define a username.
#### `update_delay` (default: `45s`)
How long to wait after startup before sending results to the receivers.
30-60s is usually good. Set higher for systems with a lot of checks.
#### `token` (default: `hackme`)
The token is used for remote results authentication on the `/relay` api endpoint.
Set to "" to disable remote result collection.
#### `hostname` (default: `''`)
The hostname is determined automatically.
If that's not working you can set it here.
#### `listen` (default: `127.0.0.1:5668`)
Provide an ip:port to bind the nrdc http listener to. Example usage:
```bash
curl localhost:5668/debug/vars | jq .receivers
curl localhost:5668/debug/vars | jq .results
curl localhost:5668/debug/vars | jq .servicechecks

```
#### `config_template` (default: `nrdc/nrdc.conf.erb`)
The template used to generate the top half of the nrdc config. If you just want
your own config, you can pass that in here.
#### `package_name` (default: `nrdc`)
The name of the nrdc package. Used internally for testing new packages.
#### `env_vars` (default: `[/bin:/usr/bin:/usr/lib/nagios/plugins]`)
Providing a proper environment PATH allows reference to service checks
using a relative path (or no path). You may also pass other env variables in
this list, such as `HTTP_PROXY`.
#### `servers` (default: `undef`)
This is used to pass in remote service-check master servers from hiera, or
directly into the nrdc class. This can be used in conjunction with or in leu of
the `nrdc::server` define for maximum flexibility.
#### `receivers` (default: `undef`)
This is used to pass in receivers from hiera, or directly into the nrdc class.
This can be used in conjunction with or in leu of the `nrdc::receiver` define for
maximum flexibility.
#### `servicechecks` (default: `undef`)
This is used to pass in service checks from hiera or directly into the nrdc class.
This can be used in conjunction with or in leu of the `nrdc::servicecheck` define
for maximum flexibility. This works best with hiera, but can be used without too.


## Define: `nrdc::server`

The server define is used to add new nrdc master service-check servers. These
servers (usually just one) are polled on their interval. They provide a json
value containing service checks to perform. These checks are added to the running
pool of service checks within nrdc. When using a master server, you will likely
not make many service checks in puppet.

#### `http_vars` (default: `''`)
Additional data variable to be sent to the server. When method is GET, these
are appended to the url. When method is POST, these are POSTed in the request.
#### `method` (default: `POST`)
What HTTP method to use when communicating with the URL.
#### `url` (default: `$name`)
The URL to communicate with to receive service checks.
#### `timeout` (default: `15s`)
How long to wait for the server to respond to the request.
#### `interval` (default: `15m`)
How often this server is polled.
#### `template_path` (default: `nrdc/server.erb`)
You may pass in a custom template for the server configuration. This is probably
not necessary, but allows you to pass in custom data when necessary.
#### `filename`
The filename is automatically created by replacing special characters in `$name`
and appending `.conf`. If a predictable name is needed, provided one.


## Define: `nrdc::receiver`

The receiver define is used to add new nrdc results receivers to the nrdc config
file. NRDC will be restarted if these change. You should define at least one receiver.
You can define them independently or pass them to the nrdc class; either from
hiera or as a parameter. See the examples above. This allows maximum flexibility.
Meaning you can define a default set of receivers for "everything" and then define
more receivers on a per-manifest or per-hiera-file basis.

#### `http_vars` (required)
Additional data variable to be POST'd. Usually a token and a command. See example
above.
#### `method` (default: `POST`)
What method to use when communicating with the URL.
#### `content_type` (default: `application/x-www-form-urlencoded`)
Content-Type header sent to the server. This should match the template content type.
#### `url` (default: `$name`)
The URL to POST data to.
#### `http_data_var` (default: `XMLDATA`)
The name of the HTTP data variable that identifies the POST'd data from the
`template_path` (explained below).
#### `template_file` (default: `/etc/nrdc/nrdp.template.xml`)
The data template sent to the receiver.
#### `timeout` (default: `5s`)
How long to wait for the receiver to accept the results.
#### `interval` (default: `5m`)
How often this receiver is updated.
#### `template_path` (default: `nrdc/receiver.erb`)
You may pass in a custom template for the receiver configuration. This is probably
not necessary, but allows you to pass in custom data if you really needed to.
#### `template_data` (default: `''`)
This allows you pass data from your config into the receiver template.
This is useful when using the slack template to specify a channel name.
#### `replace_newlines` (default: `\\n`)
By default newlines in service check output passed into a receiver template are
replaced with the literal characters `\\n`. This conforms to most xml and json
use-cases. If needs are different, pass in another string.
#### `filename`
The filename is automatically created by replacing special characters in `$name`
and appending `.conf`. If a predictable name is needed, provided one.


## Define: `nrdc::metrics`

This define is used to add metrics collectors to nrdc. Endpoints specified here
will receive nagios performance data, converted to metrics. Currently works with
Graphite, but more support may be added.

#### `$name`
Pass in the endpoint as `host:port` or `ip:port`. The data is split at the colon
and converted to host and port.
#### `timeout` (default: `5s`)
How long to wait for the TCP connection.
#### `interval` (default: `1m`)
How often to send metrics to the collector.
#### `template_path` (default: `nrdc/metrics.erb`)
You may pass in a custom template for the metrics configuration. This is
probably not necessary, but allows you to pass in custom data when needed.
#### `filename`
The filename is automatically created by replacing special characters in `$name`
and appending `.conf`. If a predictable name is needed, provided one.


## Define: `nrdc::servicecheck`

Perhaps the most important piece of this module. This define is one of the several
ways to create service checks inside NRDC.

#### `command` (required)
The check command to run.
#### `handler` (default: `''`)
Optional command handler to run when a service state changes to SOFT.
#### `hostname` (default: `''`)
If this service check is for a different host, specify the hostname here.
#### `service_name` (default: `$name`)
The name of the service to report to the receivers.
#### `timeout` (default: `60`)
How long to wait for the command to finish running, in seconds.
#### `interval` (default: `5`)
How often to run this service check, in minutes.
#### `retry_interval` (default: `2`)
How often to check a service (in minutes) when the state changes to SOFT.
If `interval` is set to 1, this will also be set to 1.
#### `retry` (default: `3`)
How many times to check the service before changing state to HARD.
#### `template_path` (default: `nrdc/servicecheck.erb`)
If you want to pass in a custom service template, you may. This is for special use.
#### `user` (default: `nobody`)
Specify the username that will run this service check.
