# Configures DNS forwarding on hosts that do not have public IPs.
#
# This uses unbound to forward all queries to a resolver (dns-resolver anycast)
class profile_dns::forwarder_a (
  $anycast_address   = '10.254.0.3',
  $anycast_community = 16001,
  $forwarders        = ['10.254.0.4', '10.254.0.7', '10.254.0.9', '10.254.0.21', '10.254.0.32', '10.254.0.33'],
) {

  $edns_client_subnet_zones = prefix(hiera('edns_client_subnet_zones', []), 'client-subnet-zone: ')

  class { 'twitch_unbound':
    val_log_level      => 1,
    custom_server_conf => concat([
      'unblock-lan-zones: yes',
      'insecure-lan-zones: yes',
    ], $edns_client_subnet_zones),
  }

  # forward all requests to DNS resolvers
  unbound::forward { '.':
    address => $forwarders,
  }

  $zones = hiera('twitch_dns_zones')

  # Override s3 zone records to bypass caching
  # XXX: Please read https://jira.twitch.com/browse/INC-1300
  $s3_endpoints = keys($zones['s3_zones'])
  unbound::forward { $s3_endpoints:
    address          => $forwarders,
    forward_no_cache => 'yes',
  }

  consul::service { 'unbound':
    consul_tags => [
      'forwarder',
      pick($::twitch_environment, $twitch_environment, 'development')
    ],
  }

  include twitch_anycast_health_enforcer::monitoring
  twitch_anycast_health_enforcer::service { 'authdns':
    interval   => '2s',
    rise       => 1,
    fall       => 1,
    addresses  => [$anycast_address],
    community  => $anycast_community,
    checks     => {
      '/usr/lib/nagios/plugins/check_dns -s localhost -H 127.0.0.1 -t 1' => {
        type    => 'cmd',
        timeout => '2s',
      },
    },
  }

  include bird::monitoring
}
