# Configures DNS an Unbound DNS resolver which requires a public address.
class profile_dns::resolver (
  $anycast_ipaddress = undef,   # undef here so development or staging dont accidentally include production IPs
  $anycast_community = '16007', # dns-resolver
) {

  # dns-auth Anycast IP
  $stub_address = '10.254.0.6'
  $zones        = hiera_hash('twitch_dns_zones')
  $zone_list    = unique(concat($zones['external'], $zones['internal']))
  # XXX: Legacy forwarding rules which serve to forward any internal queries
  # from baremetal to a Route53 inbound resolver, which has all associated
  # Private Zones
  # AWS Account ID: 673385534282
  $route53_inbound_resolver = hiera('twitch_aws_dns_masters')
  $edns_client_subnet_zones = prefix(hiera('edns_client_subnet_zones', []), 'client-subnet-zone: ')

  class { 'twitch_unbound':
    val_log_level      => 1,
    custom_server_conf => $edns_client_subnet_zones,
  }

  # These are the zones we're authority of, these requests goto dns-auth
  unbound::stub { $zone_list:
    address  => $stub_address,
    insecure => true,
  }

  # Forwarding rules for route53 Private zones
  unbound::forward { $zones['route53']: address => $route53_inbound_resolver }

  # Override s3 zone records to bypass caching
  # XXX: Please read https://jira.twitch.com/browse/INC-1300
  create_resources(unbound::stub, $zones['s3_zones'], {
    stub_first    => 'yes',
    stub_no_cache => 'yes',
    address       => [],
  })

  consul::service { 'unbound':
    consul_tags    => [
      'resolver',
      pick($::twitch_environment, $twitch_environment, 'development'),
    ],
    check_script   => '/usr/lib/nagios/plugins/check_dns -s localhost -H 127.0.0.1 -t 1',
    check_interval => '10s',
  }

  video_cloudwatch::logfile { 'unbound':
    file_path => '/var/log/unbound.log',
  }

  if !$::disable_anycast and $anycast_ipaddress {
    include twitch_anycast_health_enforcer::monitoring
    validate_integer($anycast_community)
    # DNS Resolvers have a health enforcer
    twitch_anycast_health_enforcer::service { 'resolver':
      interval   => '2s',
      healthy    => 'any', # used to ensure ALL checks must fail to bring down interface
      rise       => 1,
      fall       => 1,
      addresses  => flatten([$anycast_ipaddress]),
      community  => $anycast_community,
      checks     => { # all of these checks must fail in order for service to be unhealthy
        'external root check' => {
          check   => '/usr/lib/nagios/plugins/check_dig -4 -l . -T NS -H 127.0.0.1 -t 2',
          type    => 'cmd',
          timeout => '2s',
        },
        'external com check' => {
          check   => '/usr/lib/nagios/plugins/check_dig -4 -l com. -T NS -H 127.0.0.1 -t 2',
          type    => 'cmd',
          timeout => '2s',
        },
        'external org check' => {
          check   => '/usr/lib/nagios/plugins/check_dig -4 -l org. -T NS -H 127.0.0.1 -t 2',
          type    => 'cmd',
          timeout => '2s',
        },
      },
    }
  }

  if $anycast_ipaddress {
    include bird::monitoring
  }
}
