# == Class: twitch_trace_aggregator
#
# twitch_trace_aggregator installs and runs the
# release/trace/cmd/aggregate binary for aggregating trace events into
# transactions
class twitch_trace_aggregator (
  # Should be retrieved from facts:
  $consul_environment = $twitch_trace_aggregator::params::consul_environment,
  $deploy_environment = $twitch_trace_aggregator::params::deploy_environment,

  $repository         = $twitch_trace_aggregator::params::repository,
  $user               = $twitch_trace_aggregator::params::user,
  $base_directory     = $twitch_trace_aggregator::params::base_directory,
  $kinesis            = $twitch_trace_aggregator::params::kinesis,
  ) inherits twitch_trace_aggregator::params {

  validate_string($consul_environment, $deploy_environment)
  validate_re($repository, '[A-Za-z0-9_]/[A-Za-z0-9_]')
  validate_bool($kinesis)

  require courier

  consul::service { 'trace-aggregator':
    consul_tags => [$consul_environment],
  }

  # install the package via courier
  courier::install { 'trace':
    env     => $deploy_environment,
    repo    => $repository,
    basedir => $base_directory,
    require => Consul::Service['trace-aggregator'],
  }

  if $kinesis {
    kinesis_client::supervise { "trace_aggregate":
      executable       => join(
        [
         "/usr/bin/env", "GODEBUG=gctrace=1",
         "${base_directory}/trace/current/bin/aggregate",
         "-in-kinesis=true",
         "-out-kinesis-stream=trace-transactions",
         ], ' '),
      stream_name      => "trace-events",
      application_name => "trace-aggregator",
      require          => Courier::Install['trace'],
    }
  }
  else {
    daemontools::supervise { "trace_aggregate":
      daemon_dir     => '/var/lib/service',
      down           => false,
      user           => $user,
      syslog         => local3,
      enable_cgroups => true,
      wd             => '/',
      daemon         => "${base_directory}/trace/current/bin/aggregate",
      options        => join(
        [
         "-in-topic=\"trace-events\"",
         "-out-kinesis-stream=\"trace-transactions\"",
         "-zk=\"trace-zk.prod.us-west2.justin.tv:2181\"",
         "-kafka=\"trace-kafka.prod.us-west2.justin.tv:9092\"",
         "-kafka-group=\"trace-aggregator\"",
         '2>&1',
         ],' '),
       predaemon      => [
                          "export GOMAXPROCS=$::processorcount"
                          ],
       require        => Courier::Install['trace'],
    }
  }
}
