# == Class: twitch_trace_report
#
# twitch_trace_report installs and runs the release/trace codebase for viewing
# distributed callgraphs
#
class twitch_trace_report (
  $consul_environment = $twitch_trace_report::params::consul_environment,
  $deploy_environment = $twitch_trace_report::params::deploy_environment,
  $repository         = $twitch_trace_report::params::repository,
  $user               = $twitch_trace_report::params::user,
  $trace_data_dir     = $twitch_trace_report::params::trace_data_dir,
  $trace_reports_dir  = $twitch_trace_report::params::trace_reports_dir,
) inherits twitch_trace_report::params {
  include twitch_trace_report::web

  validate_string($consul_environment, $deploy_environment)
  validate_re($repository, '[A-Za-z0-9_]/[A-Za-z0-9_]')

  # require the courier module
  require courier

  # register the service in consul
  consul::service { 'trace_report':
    consul_tags => [$consul_environment],
  }

  # install the package via courier
  courier::install { 'trace':
    env     => $deploy_environment,
    repo    => $repository,
    basedir => $base_directory,
    require => Consul::Service['trace_report'],
  }

  file { $trace_data_dir:
    ensure => directory,
    owner  => $user,
    group  => 'root',
    mode   => '0755',
  }

  file { $trace_reports_dir:
    ensure => directory,
    owner  => $user,
    group  => 'root',
    mode   => '0755',
  }

  daemontools::supervise { "trace_cleanup":
    daemon_dir     => '/var/lib/service',
    down           => false,
    user           => $user,
    syslog         => local3,
    enable_cgroups => true,
    wd             => '/',
    daemon         => "${base_directory}/trace/current/bin/cleanup_txreport",
    options        => join([
      "-reports=\"${trace_reports_dir}\"",
      '2>&1',
    ],' '),
    predaemon      => [
      "export GODEBUG=gctrace=1",
    ],
    require        => Courier::Install['trace'],
  }

  daemontools::supervise { "trace_runreport":
    daemon_dir     => '/var/lib/service',
    down           => false,
    user           => $user,
    syslog         => local3,
    enable_cgroups => true,
    wd             => '/',
    daemon         => "${base_directory}/trace/current/bin/runtxreport",
    options        => join([
      "-reports=\"${trace_reports_dir}\"",
      "-cmd=\"${base_directory}/trace/current/bin/txreport\"",
      '2>&1',
    ],' '),
    predaemon      => [
      "export GODEBUG=gctrace=1",
    ],
    require        => Courier::Install['trace'],
  }
}
