# == Class: twitch_trace_statsdsink
#
# twitch_trace_statsdsink installs and runs the
# release/trace/cmd/statsdsink binary for emitting statsd metrics from
# trace events
class twitch_trace_statsdsink (
  $consul_environment = $twitch_trace_statsdsink::params::consul_environment,
  $deploy_environment = $twitch_trace_statsdsink::params::deploy_environment,
  $repository         = $twitch_trace_statsdsink::params::repository,
  $user               = $twitch_trace_statsdsink::params::user,
  $base_directory     = $twitch_trace_statsdsink::params::base_directory,
  $kinesis            = $twitch_trace_statsdsink::params::kinesis,
  ) inherits twitch_trace_statsdsink::params {

  validate_string($consul_environment, $deploy_environment)
  validate_re($repository, '[A-Za-z0-9_]/[A-Za-z0-9_]')
  validate_bool($kinesis)

  require courier

  consul::service { 'trace-statsdsink':
    consul_tags => [$consul_environment],
  }

  # install the package via courier
  courier::install { 'trace':
    env     => $deploy_environment,
    repo    => $repository,
    basedir => $base_directory,
    require => Consul::Service['trace-statsdsink'],
  }

  if $kinesis {
    kinesis_client::supervise { "trace_statsdsink":
      executable       => join(
        [
         "/usr/bin/env", "GODEBUG=gctrace=1",
         "${base_directory}/trace/current/bin/statsdsink",
         "-in-kinesis=true",
         "-out=statsd.internal.justin.tv:8125",
         ], ' '),
      stream_name      => "trace-events",
      application_name => "trace-statsdsink",
      require          => Courier::Install['trace'],
    }
  }
  else {
    daemontools::supervise { "trace_statsdsink":
      daemon_dir     => '/var/lib/service',
      down           => false,
      user           => $user,
      syslog         => local3,
      enable_cgroups => true,
      wd             => '/',
      daemon         => "${base_directory}/trace/current/bin/statsdsink",
      options        => join(
        [
         "-in-topic=\"trace-events\"",
         "-out=\"statsd.internal.justin.tv:8125\"",
         "-zk=\"trace-zk.prod.us-west2.justin.tv:2181\"",
         "-kafka-group=\"trace-statsdsink\"",
         "-nread=1",
         "-nprocess=4",
         "-nwrite=2",
         '2>&1',
      ],' '),
      predaemon      => [
        "export GODEBUG=gctrace=1",
      ],
      require        => Courier::Install['trace'],
    }
  }
}
