require 'fileutils'

def module_cmd(subcmd)
  "puppet module --modulepath modules #{subcmd}"
end

namespace :module do
  desc "Create a new Puppet module skeleton"
  task :new, [:module_name] do |t, arg|
    # if we have a valid module name
    if arg.module_name =~ /^[\w_]+$/
      module_dir = "modules/#{arg.module_name}"
      base_dirs = %w(manifests files templates)
      dirs = base_dirs.map do |d|
        "#{module_dir}/#{d}"
      end
      # if the base module dir exists, lets not continue
      if not File.exists?(module_dir)
        puts "Creating directories: modules/#{arg.module_name}/{#{base_dirs.join(',')}}"
        FileUtils.mkdir_p(dirs)
        # if the base manifest files dont exist, create
        %w(init.pp params.pp).each do |filename|
          manifest_file = "#{module_dir}/manifests/#{filename}"
          if not File.exists?(manifest_file)
            case filename
            when 'params.pp'
              class_name = "#{arg.module_name}::params"
            when 'init.pp'
              class_name = "#{arg.module_name} (\n) inherits #{arg.module_name}::params"
            end
            file_skel =<<-"EOS"
            class #{class_name} {
            }
            EOS
            File.open(manifest_file, 'w') do |f|
              puts "Creating manifest file: #{manifest_file}"
              f.write(file_skel.gsub(/^\s*/, ''))
            end
          end
        end
      else
        puts "Module directory exists, not continuing"
      end
    end
  end

  desc "Check existing OSS modules for changes"
  task :check_all do |t|
    puts "Checking all OSS modules for changes..."
    output = `#{module_cmd("list | grep -v '???'| awk '{print $2}'")}`.split
    module_list = output.map { |m| m.split('-').last }
    Dir.chdir($modulepath)
    module_list.each do |mod|
      changes = `#{module_cmd("changes #{mod}")}`.chomp
      puts "## #{mod}:"
      puts changes
      puts
    end
  end

  desc "Install Module from Puppet Forge"
  task :install, [:module_name] do |t,arg|
    force = ENV['FORCE'].nil? ? '' : '--force '
    sh module_cmd("install #{force}#{arg.module_name}")
  end

  desc "Upgrade Module from Puppet Forge"
  task :upgrade, [:module_name] do |t,arg|
    force = ENV['FORCE'].nil? ? '' : '--force '
    sh module_cmd("upgrade #{force}#{arg.module_name}")
  end

end
