class rubick (
  $ensure      = $rubick::params::ensure,
  $deploy_env  = $rubick::params::deploy_env,
  $canary      = $rubick::params::canary,
  $service     = $rubick::params::service,
  $conf_dir    = $rubick::params::conf_dir,
  $daemonize   = $rubick::params::daemonize,
  $pid_dir     = $rubick::params::pid_dir,
  $bind        = $rubick::params::bind,
  $port        = $rubick::params::port,
  $log_level   = $rubick::params::log_level,
  $log_dir     = $rubick::params::log_dir,
  $db_dir      = $rubick::params::db_dir,
  $cluster_dir = $rubick::params::cluster_dir,
  $masters     = $rubick::params::masters,
  $replicas    = $rubick::params::replicas,
  $sandstorm   = $rubick::params::sandstorm,
  $role        = $rubick::params::role,
  $region      = $rubick::params::region,
  $host        = $rubick::params::host,
) inherits rubick::params {
  validate_re($ensure, [ '^present$', '^absent$' ])
  validate_absolute_path([ $conf_dir, $pid_dir, $log_dir, $db_dir, $cluster_dir ])
  validate_ip_private($bind)
  validate_string($service)

  $conf_file    = "${service}.conf"
  $pid_file     = "${service}.pid"
  $log_file     = "${service}.log"
  $db_file      = "${service}.rdb"
  $cluster_file = "${service}-nodes.conf"

  $dir_ensure = $ensure ? {
    'absent' => 'absent',
    default => 'directory',
  }
  # TODO Handle duplicate values properly.
  file { [ $conf_dir, $db_dir, $log_dir ]: # $cluster_dir is a duplicate
    ensure => $dir_ensure,
    mode   => '0755',
    owner  => 'redis',
    group  => 'redis',
  }

  $conf_path = "${conf_dir}/${conf_file}"
  $env = $::twitch_environment

  twitch_sandstorm_agent::template { "rubick.conf":
    destination     => $conf_path,
    ensure          => $ensure,
    owner           => 'redis',
    group           => 'redis',
    mode            => '0400',
    restart_command => "/usr/bin/svc -h /etc/service/${service}",
    contents        => template("rubick/rubick.conf.erb"),
  }

  twitch::service { $service:
    env                   => $deploy_env,
    canary                => $canary,
    daemontools_ensure    => $ensure,
    service_repo          => 'video/rubick',
    user                  => 'redis',
    daemon_cmd            => "./rubick ${conf_path}",
    predaemon             => [
      "systemctl is-active --quiet video-milliner",
      "source /etc/milliner-client-env.sh",
      "mkdir -p ${pid_dir} && chown redis:redis ${pid_dir}",
      "test -f /etc/profile.d/proxy.sh && source /etc/profile.d/proxy.sh"
    ],
    consul_advertise_port => $port,
    require               => Class['milliner'],
    codedeploy_compat     => true,
  }

  logrotate::rule{ 'rubick':
    ensure        => $ensure,
    path          => "${log_dir}/${service}-*.log",
    rotate_every  => 'day',
    rotate        => 14,
    copytruncate  => true,
    delaycompress => true,
    compress      => true,
    ifempty       => false,
    missingok     => true,
    su            => true,
    su_owner      => 'redis',
    su_group      => 'redis',
  }

  # Install our simple script that lists all nodes in a redis cluster.
  # This is required for our health check and diamond config generation.
  # We put this in /usr/local/bin because it's useful enough to have in $PATH.
  $redis_cluster_nodes = "/usr/local/bin/redis-cluster-nodes"
  file { $redis_cluster_nodes:
    ensure => $ensure,
    mode   => '0755',
    source => "puppet:///modules/${module_name}/redis-cluster-nodes",
  }

  # Run the rubick health check.
  twitch_nrdp::check { "check_redis_cluster_${port}":
    ensure       => $ensure,
    service_name => "redis-cluster-check ${port}",
    command      => "check_redis_cluster -l -p ${port}",
    interval     => 1,
    require      => File[$redis_cluster_nodes],
  }

  # Do metric collection for each possible redis port.
  $instances = range($port, $port + $masters + $replicas - 1)

  twitch_redis::server_metrics { $instances:
    ensure => $ensure,
  }
}
