# sgauge::reservoir is a tool that aggregates, and processes sFlow data.
#
# This data is produced by the routers channeled via canal and then fed into
# the video system.
#
# pool_id is not used yet, just adding for the future and to allow assigning
# the pool from hiera.
class sgauge::reservoir (
  $deploy_environment = pick($::twitch_environment, $twitch_environment, 'production'),
  $admin_port         = 6061,
  $app_dir            = '/opt/twitch/reservoir',
  $canary             = false,
  $statsd_environment = undef,
  $inbound_handlers   = 50,
  $log_level          = 'info',
  $statsd_host_port   = 'localhost:8125', # This should be in a central default.yaml location
  $user               = 'nobody',
  $pool_id            = '',
  $config_location    = '/opt/twitch/reservoir/config.json',
  $memory_percent     = 80,
  $prod_communities   = [],
) {
  # Validate that memory_percent is between 0-99
  # Despite what puppet-lint thinks the below key has to be wrapped as a string
  # in order to pass it to validate_re.
  validate_re($memory_percent, '^[0-9][0-9]?$')
  validate_re($admin_port, '^[0-9]+$')
  validate_re($inbound_handlers, '^[0-9]+$')

  if $statsd_environment == undef {
    $real_statsd_environment = $deploy_environment
  } else {
    $real_statsd_environment = $statsd_environment
  }

  include sgauge
  validate_string($deploy_environment,$admin_port,$pool_id)

  # If a pool_id is specified then we want to add the '-'
  if $pool_id != '' {
    $real_pool_id = "-${pool_id}"
  }

  envoy::service { 'reservoir':
    port => $admin_port,
  }

  # By default values get passed in by puppet they end up being strings, even
  # if they could otherwise just be integers. This means that values such as
  # $admin_port end up coming out in the final JSON looking like `"admin_port":
  # "9000"` this in turn causes reservoir to choke since it is expecting an int
  # and not a string. By adding `0 + $foo` this forces puppet to coerce the
  # internal type from a string to an int.

  $config = delete_undef_values({
    'production_communities' => $prod_communities,
    'admin_port'             => 0 + $admin_port, # See comment above
    'log_level'              => $log_level,
    'hostname'               => $::hostname,
    'environment'            => $real_statsd_environment,
    'statsd_host'            => $statsd_host_port,
    'inbound_handlers'       => 0 + $inbound_handlers, # See comment above
  })

  file { $config_location:
      mode    => '0644',
      owner   => 'root',
      group   => 'root',
      content => pretty_json($config),
      notify  => Daemontools::Supervise['reservoir'],
  }

  $memory_limit = floor(($::memorysize_mb * 1024.0) * ($memory_percent / 100.0))

  twitch::service { 'reservoir':
    env          => "${deploy_environment}${real_pool_id}",
    daemon_cmd   => "./reservoir -config \"${config_location}\"",
    predaemon    => [
      "ulimit -v ${memory_limit}",
      "ulimit -m ${memory_limit}",
      '. /etc/profile.d/proxy.sh',
      'if ! systemctl is-active --quiet video-milliner; then',
      '    echo "milliner is not up yet"',
      '    sleep 1',
      '    exit 1',
      'fi',
      '. /etc/milliner-client-env.sh'
    ],
    service_repo => 'video/reservoir',
    user         => $user,
  }
}
