class twitch::haproxy::frontends::usher_external (
  $instance_name       = 'usher_external',
  $frontend_http_name  = 'usher_external_frontend_http',
  $frontend_https_name = 'usher_external_frontend_https',
  $frontend_tcp_name   = 'usher_external_frontend_tcp',
  $vizima_cert_prefix  = "video/usher-proxy/${::twitch_environment}/vizima_twitch_tv",
  $ssl_cert_prefix     = "video/usher-proxy/${::twitch_environment}/cert_usher_ttvnw_net",
  $tls_keys_path       = '/etc/ssl/private/usher.ttvnw.net-key+cert+intermediate.pem.tls_keys',
  $cert_prefix         = "video/usher-external/${::twitch_environment}/cert_usher_ttvnw_net",
  $default_backend     = 'regional_mmr_backend',
  $processes           = $::processorcount / 2,
  $use_so_reuseport    = true,
  $cert_type           = 'rsa',
  $settings            = {
    'default_backend' => $default_backend,
    'maxconn'         => '52000',
    'timeout'         => [
                          'connect 5000',
                          'client 20000',
                          'server 20000',
    ],
  }
){
  $ssl_bind_options = ["ssl no-sslv3 crt /etc/haproxy/ssl tls-ticket-keys ${tls_keys_path}"]

  $http_opts = {
    'mode'          => 'http',
    'log'           => 'global',
    'option'        => [
                        'httplog',
                    ],
    'http-request'  => [
                        'set-header X-Real-IP %[src]',
                        'set-header X-Forwarded-Proto https if { ssl_fc }',
                        'set-header X-Real-Host %[req.hdr(Host)]',
                        "set-header X-Source-Cluster ${::pop}",
                        'del-header X-Forwarded-For'
                    ],
    'acl'           => 'use_regional_mmr path_beg /pathfinder',
    'use_backend'   => 'regional_mmr_backend if use_regional_mmr',
    'http-response' => ['set-header Cache-Control "no-cache,no-store"','del-header via'],
    'log-format'    => '%ci:%cp\ [%t]\ %ft\ %b/%s\ %Tq/%Tw/%Tc/%Tr/%Tt\ %ST\ %B\ %CC\ %CS\ %tsc\ %ac/%fc/%bc/%sc/%rc\ %sq/%bq\ %hr\ %hs\ %{+Q}r\ %sslc\ %sslv',
  }

  $http_settings = deep_merge($settings, $http_opts)

  twitch_haproxy::frontend { $frontend_https_name:
    instance_name    => $instance_name,
    ports            => [443],
    ipaddress        => '*',
    options          => $http_settings,
    nbproc           => $processes,
    use_so_reuseport => $use_so_reuseport,
    bind_options     => $ssl_bind_options,
    require          => Twitch_haproxy::Ssl_certificate['usher_ttvnw_cert'],
  }

  twitch_haproxy::frontend { $frontend_http_name:
    instance_name    => $instance_name,
    ports            => [80],
    ipaddress        => '*',
    options          => $http_settings,
    nbproc           => $processes,
    use_so_reuseport => $use_so_reuseport,
  }

  # WARNING: do not gaze into the void below here.

  twitch_sandstorm_agent::template { 'usher_ttvnw_tls_keys':
    destination     => $tls_keys_path,
    contents        => template("${module_name}/haproxy/services/tls_keys.erb"),
    owner           => 'root',
    group           => 'ssl-cert',
    mode            => '0640',
    restart_command => "/etc/init.d/haproxy_${instance_name} tls-key-reload",
  }


  twitch_ssl_certificate::certificate { 'staging_vse_live-video-ext_a2z_com':
    ensure               => $::twitch_environment ? { 'staging' => 'present', default => 'absent' },
    cert_filesystem_path => '/etc/haproxy/ssl/staging_vse_live-video-ext_a2z_com.rsa',
    cert_contents        => join([
      "{{ key \"video/usher-proxy/staging/staging_vse_live-video-ext_a2z_com/cert\" }}",
      "{{ key \"video/usher-proxy/staging/staging_vse_live-video-ext_a2z_com/intermediate\" }}",
      "{{ key \"video/usher-proxy/staging/staging_vse_live-video-ext_a2z_com/key\" }}"
    ], "\n"),
    cert_restart_cmd     => "/bin/systemctl reload haproxy_${instance_name}",
  }

  twitch_haproxy::ssl_certificate { 'usher_ttvnw_cert':
    instance_name        => $instance_name,
    sandstorm_key_prefix => $ssl_cert_prefix,
    alg_type             => $cert_type,
  }

  # These three regions, us-west-2, us-east-2, and eu-west-1 have staging and prod wildcart certs for IVS.
  if ($::twitch_environment == 'production') {
    twitch_haproxy::ssl_certificate { 'wildcard_us_west_2_playback_live_video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => 'video/usher-proxy/production/wildcard_us-west-2_playback_live-video_net',
      alg_type             => $cert_type,
    }

    twitch_haproxy::ssl_certificate { 'wildcard_us-east-1_playback_live-video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => 'video/usher-proxy/production/wildcard_us-east-1_playback_live-video_net',
      alg_type             => $cert_type,
    }

    twitch_haproxy::ssl_certificate { 'wildcard_eu-west-1_playback_live-video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => 'video/usher-proxy/production/wildcard_eu-west-1_playback_live-video_net',
      alg_type             => $cert_type,
    }
  # if twitch_environment is not production, assume staging  
  } else {
      twitch_haproxy::ssl_certificate { 'wildcard_us_west_2_staging_playback_live_video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => 'video/usher-proxy/staging/wildcard_us-west-2_staging_playback_live-video_net',
      alg_type             => $cert_type,
    }

    twitch_haproxy::ssl_certificate { 'wildcard_us-east-1_staging_playback_live-video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => 'video/usher-proxy/staging/wildcard_us-east-1_staging_playback_live-video_net',
      alg_type             => $cert_type,
    }

    twitch_haproxy::ssl_certificate { 'wildcard_eu-west-1_staging_playback_live-video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => 'video/usher-proxy/staging/wildcard_eu-west-1_staging_playback_live-video_net',
      alg_type             => $cert_type,
    }
  }
  # There seem to be some serious issues with Sandstorm trying to load secrets that don't exist.
  # There aren't staging certs for these regions so these were only added to "prod" to prevent issues with staging.
  # The older ones weren't touched to prevent any kind of breakage if there's a dependency on those ones somewhere.
  if ($::twitch_environment == "production") {
    twitch_haproxy::ssl_certificate { 'wildcard_ap-northeast-1_playback_live-video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => "video/usher-proxy/${::twitch_environment}/wildcard_ap-northeast-1_playback_live-video_net",
      alg_type             => $cert_type,
    }

    twitch_haproxy::ssl_certificate { 'wildcard_ap-northeast-2_playback_live-video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => "video/usher-proxy/${::twitch_environment}/wildcard_ap-northeast-2_playback_live-video_net",
      alg_type             => $cert_type,
    }

    twitch_haproxy::ssl_certificate { 'wildcard_ap-south-1_playback_live-video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => "video/usher-proxy/${::twitch_environment}/wildcard_ap-south-1_playback_live-video_net",
      alg_type             => $cert_type,
    }

    twitch_haproxy::ssl_certificate { 'wildcard_eu-central-1_playback_live-video_net_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => "video/usher-proxy/${::twitch_environment}/wildcard_eu-central-1_playback_live-video_net",
      alg_type             => $cert_type,
    }
  }

  twitch_haproxy::ssl_certificate { 'vizima_twitch_tv_cert':
    instance_name        => $instance_name,
    sandstorm_key_prefix => $vizima_cert_prefix,
    alg_type             => $cert_type,
  }
}
