class twitch::haproxy::frontends::video_weaver_ssl (
  $instance_name    = 'video_weaver_ssl',
  $frontend_name    = 'https-keepalive-rsa-http',
  $cert_prefix      = "video/video-edge/production/${::pop}/wildcard_hls_ttvnw_net",
  $default_backend  = 'https-keepalive',
  $use_so_reuseport = true,
  $nbproc           = 1,
  $enable_abs       = false,
  $enable_ivs       = false,
){
  include twitch_nrdp

  # the default pop ssl cert for video edge
  twitch_haproxy::ssl_certificate { 'pop_ssl_cert':
    instance_name        => $instance_name,
    sandstorm_key_prefix => $cert_prefix,
    before               => Twitch_haproxy::Frontend[$frontend_name],
  }

  if str2bool($enable_ivs) {
    # *.pop.hls.live-video.net certificate on edge
    twitch_haproxy::ssl_certificate { 'pop_ivs_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => "video/video-edge/production/${::pop}/wildcard_hls_live-video_net",
      before               => Twitch_haproxy::Frontend[$frontend_name],
    }
  }

  if ($enable_abs) {
    # certificate for ABS streams on edge
    twitch_haproxy::ssl_certificate { 'pop_abs_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => "video/video-edge/production/${::pop}/wildcard_abs_hls_ttvnw_net",
      before               => Twitch_haproxy::Frontend[$frontend_name],
    }

    # certificate for Non ABS streams on edge
    twitch_haproxy::ssl_certificate { 'pop_no-abs_cert':
      instance_name        => $instance_name,
      sandstorm_key_prefix => "video/video-edge/production/${::pop}/wildcard_no-abs_hls_ttvnw_net",
      before               => Twitch_haproxy::Frontend[$frontend_name],
    }
  }

  # key needs to *not* in the same directory sourced cert files
  $tls_keys_path        = "/etc/ssl/private/wildcard.${::pop}.hls.ttvnw.net-key+cert+intermediate.pem.tls_keys"
  twitch_sandstorm_agent::template { 'pop_tls_keys':
    destination     => $tls_keys_path,
    contents        => template("${module_name}/haproxy/services/tls_keys.erb"),
    owner           => 'root',
    group           => 'ssl-cert',
    mode            => '0640',
    restart_command => "/etc/init.d/haproxy_${instance_name} tls-key-reload",
    before          => Twitch_haproxy::Frontend[$frontend_name],
  }

  # Add sandstorm checks to this sandstormified-host.
  realize(Twitch_nrdp::Check['check_sandstorm_agent'])
  realize(Twitch_nrdp::Check['check_sandstorm_agent_ops'])

  twitch_haproxy::frontend { $frontend_name:
    instance_name    => $instance_name,
    ports            => [443],
    bind_options     => ["ssl no-sslv3 crt /etc/haproxy/ssl tls-ticket-keys ${tls_keys_path}"],
    ipaddress        => '*',
    options          => {
      'default_backend' => $default_backend,
      'option'          => [
        'http-keep-alive',
        'forwardfor except 127.0.0.0/8 header X-Client-IP',
      ],
      'monitor-uri'     => '/health',
      'monitor fail'    => 'if no_srv',
      'http-request'    => [
        'del-header X-Client-IP',
        'set-header Proxied-HTTPS true',
      ],
      'acl'             => [
        "no_srv nbsrv(${default_backend}) lt 1",
      ],
    },
    use_so_reuseport => $use_so_reuseport,
    nbproc           => $nbproc,
  }
}
