class twitch::haproxy::usher_external (
  $instance_name    = 'usher_external',
  $stats_port       = 2012,
  $http2            = false,
  $use_cloudwatch   = true,
  $consul_service   = 'production.usher-api-origin@sjc02',
  $backend_options  = 'check inter 1000 maxconn 200',
  $global_options   = {
    'nbproc'                    => $::processorcount / 2,
    'user'                      => 'www-data',
    'group'                     => 'www-data',
    'daemon'                    => '',
    'tune.ssl.default-dh-param' => '2048',
    'ssl-default-bind-ciphers'  => hiera('ciphers::performance'),
    'ssl-default-bind-options'  => 'no-sslv3',
    'tune.ssl.cachesize'        => '80000',
    'tune.ssl.lifetime'         => '14400',
    'log'                       => '127.0.0.1 local3',
    'log-tag'                   => 'haproxy_usher',
    'pidfile'                   => '/var/run/haproxy-usher_external.pid'
  },
  $defaults_options = {
    'mode'             => 'http',
    'unique-id-format' => $::pop,
    'option'           => [
      'dontlog-normal',
      'dontlognull',
      'tcpka',
      'forwardfor except 127.0.0.0/8'
    ],
    'timeout'          => ['connect 10s', 'client 20s', 'server 20s'],
    'log'              => 'global',
  },
) {

  # We attempt to remap the regions we don't have MMR Proxies in. In the event we're missing a mapping,
  # this will fall back to proxying to PDX. We should keep an eye on this to make sure that it works
  # correctly
  if $::twitch_environment == "staging" {
    # there are only us-east-2 and eu-west-2 mappings for gamma
    $_mmr_region = $::default_region ? {
      'eu-west-1' => 'eu-west-2',
      'eu-west-2' => 'eu-west-2',
      default     => 'us-west-2',
    }

    $_mmr_stage = "gamma"
  } else {
    $_mmr_region = $::default_region ? {
      'us-east-1' => 'us-east-2',
      'us-east-2' => 'us-east-2',
      'eu-west-1' => 'eu-west-2',
      'eu-west-2' => 'eu-west-2',
      default     => 'us-west-2',
    }

    $_mmr_stage = "prod"
  }

  validate_bool(str2bool($use_cloudwatch))

  class { 'twitch::haproxy::backends::usher_mmr_backend':
    instname      => $instance_name,
    primary_url   => "${::twitch_environment}.mmr-external.live-video.a2z.com",
    secondary_url => "${::twitch_environment}-canary.mmr-external.live-video.a2z.com",
    regional_url  => "${_mmr_region}.${_mmr_stage}.mmrproxy.live-video.a2z.com",
    remote_port   => 80,
  }

  class { 'twitch::haproxy::frontends::usher_external':
    instance_name    => $instance_name,
    default_backend  => 'regional_mmr_backend',
    use_so_reuseport => true,
  }

  twitch_haproxy::instance { $instance_name:
    defaults_options     => $defaults_options,
    global_options       => $global_options,
    stats_port           => $stats_port,
    create_stats_sockets => true,
    with_consul_template => false,
  }

  if (str2bool($use_cloudwatch)) {
    $_use_cloudwatch_ensure = 'present'
  } else {
    $_use_cloudwatch_ensure = 'absent'
  }

  twitch_svclogs::logfile { 'haproxy-error':
    ensure         => $_use_cloudwatch_ensure,
    log_group_name => "/twitch/${::pop}/haproxy",
    filename       => 'haproxy_usher.log',
    use_milliner   => true,
  }

  # This installs Nagios passive checks onto servers that have secrets to monitor.
  realize(Twitch_nrdp::Check['check_sandstorm_agent', 'check_sandstorm_agent_ops'])
  twitch_nrdp::check { 'video-service-ext_haproxy_stats':
    command  =>
      "check_haproxy_stats.py -s '/var/run/' -g 'haproxy-usher_external-stats.*.sock' -c conn-ratio sess-ratio available-backend --crit_ratio 0.96"
    ,
    interval => 1,
    retry    => 3,
  }
}
