define twitch::service(
  $daemon_cmd,
  $predaemon,
  $service_repo,
  $ensure                = 'present',
  $daemontools_ensure    = 'present',
  $env                   = pick($::twitch_environment, $twitch_environment, 'production'),
  $canary                = false,
  $service               = $name,
  $manage_service        = undef,
  $concurrency           = 0,
  $user                  = $name,
  $group                 = undef,
  $base_dir              = hiera('twitch_basedir'),
  $app_dir               = undef,
  $syslog                = 'local3',
  $consul_check_script   = undef,
  $consul_check_http     = undef,
  $consul_check_interval = undef,
  $consul_advertise_port = undef,
  $enable_cgroups        = false,
  $cgroups_basedir       = '/sys/fs/cgroup',
  $cgroups_rules         = {},
  $cgroups_namespace     = 'service',
  $cgroups               = [
  'cpuacct',
  'memory',
  ],
  $monit_ensure          = 'absent',
  $monit_check_port      = 80,
  $monit_check_path      = '/health',
  $monit_check_interval  = 30,
  $monit_check_tries     = 3,
  $monit_check_cycles    = 5,
  $monit_check_timeout   = 5,
  $monit_check_mode      = 'http_code',  # or 'http_cx', or 'tcp_cx'
  $monit_down_kill       = false,
  $monit_alert_email     = undef,
  $svclogs               = false,
  $svclogs_account       = 'twitch-video',
  $svclogs_region        = 'us-west-2',
  $svclogs_use_milliner  = false,
  $courier_install       = true,
  $codedeploy_compat     = false,

) {
  validate_re($monit_check_mode, '(^http_code$|^http_cx$|^tcp_cx$)',
    "${monit_check_mode} is not permitted for monit_check_mode. Allowed check modes are 'http_mode', 'http_cx', or 'tcp_cx'.")
  validate_bool($monit_down_kill)
  validate_bool($courier_install)

  if ($app_dir) {
    $real_app_dir = $app_dir
  } else {
    $real_app_dir = "${base_dir}/${name}"
  }

  validate_absolute_path($real_app_dir)

  # for legacy dirty/clean environment
  # automating this will ensure consistency from now forward.
  $deploy_env = $service ? {
    'akamai_stats'            => "clean-${env}",
    'canal'                   => "clean-${env}",
    'edgecast_stats'          => "clean-${env}",
    'fastly_stats'            => "clean-${env}",
    'hose_stats'              => "clean-${env}",
    'hosesapi'                => "clean-${env}-physical",
    'reservoir'               => "clean-${env}",
    'rtqos_kinesis_consumer'  => "clean-${env}",
    'twitch_beastmaster'      => "clean-${env}",
    'twitch_wall'             => "clean-${env}",
    'varnish-to-spade'        => "clean-${env}",
    default                   => $env,
  }

  # allow a flag for canary environments.
  # for snowflakey additional environments, also allow specifying a name.
  $consul_tags = $canary ? {
    true    => [ $deploy_env, 'canary' ],
    false   => [ $deploy_env ],
    default => [ $deploy_env, $canary ],
  }

  # don't try to create 'root' as local user.
  if ( $user != 'root' ) {
    $dt_user = $user
    if !defined(Twitch::Local_user[$user]) {
      # create service user/group
      twitch::local_user { $user:
        group => $group,
      }
    }
  } else { # user = root
    $dt_user = undef
  }

  # install/deploy payload
  # TODO: this courier install needs to use jtv because upstream deploy tools
  #       change uids to jtv explicitly when deploying, so we need file perms correct
  if $courier_install {
    courier::install { $service:
      ensure            => $ensure,
      repo              => $service_repo,
      env               => $deploy_env,
      basedir           => $base_dir,
      user              => 'jtv',
      group             => 'jtv',
      codedeploy_compat => $codedeploy_compat,
      before            => Daemontools::Supervise[$service],
    }
  }

  $dt_ensure = $ensure ? {
    'absent' => 'absent',
    default  => $daemontools_ensure
  }

  # setup daemontools
  daemontools::supervise { $service:
    ensure            => $dt_ensure,
    daemon_dir        => '/var/lib/service',
    daemon            => "${daemon_cmd} 2>&1",
    wd                => "${real_app_dir}/current",
    syslog            => $syslog,
    user              => $dt_user,
    env               => $env,
    concurrency       => $concurrency,
    manage_service    => $manage_service,
    predaemon         => flatten([$predaemon]),
    enable_cgroups    => $enable_cgroups,
    cgroups_basedir   => $cgroups_basedir,
    cgroups_rules     => $cgroups_rules,
    cgroups_namespace => $cgroups_namespace,
    cgroups           => $cgroups,
  }

  # run a monit 'service' that opens a tcp/http connection, and tells
  # supervise to restart svc if an error is returned.
  #
  # supported monit_check_modes:
  #  - http_code: (default) hits http://localhost:${monit_check_port}/${monit_check_path}
  #       Success is http response code less than 400
  #  - http_cx: Hits same endpoint as http_code. Success on _any_ response.
  #  - tcp_cx: Establish TCP cxn to localhost:${monit_check_port}.
  #
  # ** monit will restart services downed with 'svc -d'. beware! **
  # if you want to down a monit-watched service: 'monit unmonitor servicename'.
  # once we're at vers >= 5.18, we can use 'onreboot nostart'.

  $http_code_check = "if failed port ${monit_check_port} proto http request ${monit_check_path} status lt 400 then restart"
  $http_cx_check   = "if failed port ${monit_check_port} proto http request ${monit_check_path} status gt 1 then restart"
  $tcp_cx_check    = "if failed port ${monit_check_port} timeout ${monit_check_timeout} seconds then restart"

  $check_limits    = "if ${monit_check_tries} restarts within ${monit_check_cycles} cycles then unmonitor"

  $checks = $monit_check_mode ? {
    'tcp_cx'  => $tcp_cx_check,
    'http_cx' => $http_cx_check,
    default   => $http_code_check,
  }

  $down_arg = $monit_down_kill ? {
    true  => '-dk',
    false => '-d',
  }

  $custom_checks = flatten([$checks, $check_limits])

  twitch_monit::check { $service:
    ensure         => $monit_ensure,
    regex          => $daemon_cmd,
    uid            => 'root',
    cwd            => '/var/tmp',
    start_program  => "/usr/bin/svc -u /var/lib/service/${service}",
    stop_program   => "/usr/bin/svc ${down_arg} /var/lib/service/${service}",
    custom_checks  => $custom_checks,
    check_interval => $monit_check_interval,
    alert_email    => $monit_alert_email,
    require        => Daemontools::Supervise[$service],
  }

  # advertise service
  consul::service { $service:
    ensure         => $ensure,
    port           => $consul_advertise_port,
    consul_tags    => $consul_tags,
    check_script   => $consul_check_script,
    check_http     => $consul_check_http,
    check_interval => $consul_check_interval,
    require        => Daemontools::Supervise[$service],
  }

  if ($svclogs) {
    # ingest the log for this service into cloudwatch logs.
    twitch_svclogs::logfile { $service:
      ensure       => $ensure,
      account      => $svclogs_account,
      region       => $svclogs_region,
      use_milliner => $svclogs_use_milliner,
    }
  }

  notify { "${service} - ${service_repo} - ${deploy_env}": }

}
