#!/bin/bash

SVC_BIN=$(which svc)
SVC_DIR=/etc/service
USER=$(whoami)
CURRENT_DATE=$(date +"%s")
FUTURE_DATE=$(date '+%s' -d "+3 days")
HOST=$(hostname -f)
NAGIOS_SERVER=central-nagios.internal.justin.tv

function usage {

cat <<\EOF

Enables and Disables Anycast Health enforcer
This will also enable and disable downtime for DNS and Squid

 ./cool.sh -s authdns -d 

-s Anycast service you want to Enable/Disable
-c Adds a custom comment for downtime
-d Disable Health Enforcer and Anycast adr
-e Enable Health Enforcer and Anycast adr
-S List the status of all Health Enforcers
EOF

}

function verify_service {

  # verifies if the service provided exists
  if [ ! -d $enforcer_service ];then
    echo service not found
    exit 1
  fi

}

# Handles the creating and removing downtimes
function nagios_downtime {

# If the service matches a certain name then the nagios alert
# should have a standard name.

  if [ $service == authdns ];then
    nagios_check_name="dns interface and resolution"
  fi

# If there is no nagios check known to downtime then we should stop here
  if [ -z "$nagios_check_name" ];then
    echo No nagios check to downtime
    exit 0
  fi

# Grab information related to the service from live status and set variables
downtime_present=$(nc $NAGIOS_SERVER 6557 <<EOF
GET downtimes
Columns: host_name service_description id
Filter: host_name = $HOST
EOF
)

downtime_id=$(echo $downtime_present | grep "$nagios_check_name" | cut -f3 -d';')
nagios_enforcer_downtime="COMMAND [$(date +%s)] SCHEDULE_SVC_DOWNTIME;$HOST;$nagios_check_name;$CURRENT_DATE;$FUTURE_DATE;1;0;7200;$USER;$COMMENT"
nagios_remove_downtime="COMMAND [$(date +%s)] DEL_SVC_DOWNTIME;$downtime_id"

# Here is we set the downtime if we are disabling or remove it if we are enabling
  if [ ! -z "$DISABLE" ];then
    echo $downtime_present | grep "$nagios_check_name" >/dev/null
      if [ $? -eq 0 ];then
        echo downtime present for $service on $HOST
        exit 0
      fi
    echo $nagios_enforcer_downtime | nc $NAGIOS_SERVER 6557
    echo downtime added to $nagios_check_name on $HOST
  fi

  if [ ! -z "$ENABLE" ];then
    echo $nagios_remove_downtime | nc $NAGIOS_SERVER 6557
    echo downtime for $nagios_check_name removed on $HOST
  fi

exit 0
}

# checks the status of the health enforcer and anycast adr
function enforcer_status {

# If you want to know the status of one service this will tell you
  if [ ! -z "$service" ];then
    verify_service
    echo current status of $service health enforver
    enforcer_service=$SVC_DIR/anycast_health_enforcer_lo_$service/
    sudo svstat $enforcer_service
    cat $enforcer_service >/dev/null 2>&1 || echo down file present
    exit
  fi

# List all health enforcers running on the box
all_enforcers=$SVC_DIR/anycast_health_enforcer_lo_*
enforcer_list=$(sudo svstat $all_enforcers | awk '{print $1}'| cut -f1 -d':')

for i in $(echo $enforcer_list);do
enforcer_full_status=$(sudo svstat $i)
enforcer_status=$(sudo svstat $i | awk '{print $2}')
service=$(echo $i | cut -f5 -d'_')

# Check if a down file is in place
down_file=$(cat $i/down >/dev/null 2>&1)
  if [ $? -eq 0 ];then
    enforcer_down_file=present
  else
    enforcer_down_file=missing
  fi

# Check the status of the interface
ifconfig | grep $service  >/dev/null
  if [ $? -eq 0 ];then
    anycast_interface=up
  else
    anycast_interface=down
  fi

  if [ $enforcer_status == "down" ];then
    echo $enforcer_full_status down file:$enforcer_down_file anycast interface:$anycast_interface
    continue
  fi

echo $enforcer_full_status down file:$enforcer_down_file anycast interface:$anycast_interface

done
exit 0
}

# Enable Health enforcer
function enable_health_enforcer {

echo enabling $service
sudo rm $enforcer_service/down
sudo $SVC_BIN -u $enforcer_service
nagios_downtime
}

# disable health enforcer and turn off anycast adr
function disable_health_enforcer {

echo disabling $service
sudo touch $enforcer_service/down
sudo $SVC_BIN -d $enforcer_service
echo taking down interface lo:$service
sudo ifdown lo:$service
nagios_downtime
}

while getopts ":s:c:decS" opt
  do
    case $opt in
      s)
        service="$OPTARG"
        enforcer_service=$SVC_DIR/anycast_health_enforcer_lo_$service
        verify_service
      ;;
      c)
        COMMENT="$OPTARG"
      ;;
      d)
        DISABLE=true
      ;;
      e)
        ENABLE=true
      ;;
      S)
        enforcer_status
      ;;
      *) usage
      ;;
    esac
done

shift $((OPTIND-1))

  if [ -z "$COMMENT" ];then
    COMMENT="$service is being placed in maintenance mode"
  fi

  if [ -z "$service" ];then
    echo No service specified
    usage
    exit 1
  fi
    
  if [ ! -z "$DISABLE" ];then
    verify_service
    disable_health_enforcer
  fi

  if [ ! -z "$ENABLE" ];then
    verify_service
    enable_health_enforcer
  fi
