# Define: twitch_anycast::health_enforcer::daemon
#  Sets up anycast health enforcer to check the health of anycast connectivity
#
# ** Params **
# [ensure]
#  Enforce this presence for the instance. Default: 'present' Type: String
#
# [health_url]
#  URL to check service health.  Required.  Type: String
#
# [check_command]
#  Unix command to perform the status check.  Exiting zero is considered healthy, non-zero is unhealthy. Type: String
#
# [connect_only]
#  Connect to the endpoint but disregard the status.  If connected, returns healthy.  Type: Bool
#
# [source_ip]
#  When using a connect-only test, the source IP address to send the request from. Type: String
#
# [no_op]
#  Just print what would happen but do not execute.  Type: Bool
#
# [interval]
#  Time to wait between probes.  Type: String
#
# [window]
#  History to keep of backend success/fail.  Type: String
#
# [threshold]
#  Number of successes required in history to consider backend healthy.  Type: String
#
# [timeout]
#  Backend timeout limit in seconds. Type: String
#
# [alert_email]
#  Comma-separated email addresses for alert emails. Type: String
#
# [nagios_service]
#  Nagios service name to use in passive checks.  Type: String
#
# [nagios_server]
#  Nagios server to send passive checks.  Type: String
#
# [nrdp_folder]
#  Folder to write NRDP check data too. Ex: /usr/local/nrdp/checkdata  Type: String
#
define twitch_anycast::health_enforcer::daemon (
  $ensure         = 'present',
  $health_url     = undef,
  $check_command  = undef,
  $connect_only   = false,
  $source_ip      = 'localhost',
  $no_op          = false,
  $interval       = 2.0,
  $window         = 2,
  $threshold      = 1,
  $timeout        = 5,
  $alert_email    = undef,
  $nagios_service = undef,
  $nagios_server  = undef,
  $nrdp_folder    = undef,
) {
  require twitch_anycast::health_enforcer

  validate_bool($connect_only,)

  if $nrdp_folder {
    validate_absolute_path($nrdp_folder)
    include twitch_nrdp
  }

  if ($health_url == undef) and ($check_command == undef) {
    fail('twitch_anycast::health_enforcer::daemon requires either health_url or check_command.')
  }

  if ($health_url != undef) and ($check_command != undef) {
    fail('twitch_anycast::health_enforcer::daemon requires either health_url or check_command, not both.')
  }


  # split interface name from daemon name
  $intf_name   = $name
  $daemon_name = regsubst($name, ':', '_')
  $script_name = $twitch_anycast::health_enforcer::script_name
  $app_dir     = $twitch_anycast::health_enforcer::app_dir
  $virtenv_dir = $twitch_anycast::health_enforcer::virtenv_dir

  $noop_flag = $no_op ? {
    false    => '',
    default  => '--noop',
  }

  $connect_flag = $connect_only ? {
    true    => '--connect-only',
    default => '',
  }

  $real_nagios_service = $nagios_service ? {
    undef   => $nagios_service,
    default => shellquote("${nagios_service} ${name}"),
  }

  $_check_command = $check_command ? {
    undef   => $check_command,
    default => shellquote($check_command),
  }

  $daemon_args = join_keys_to_values(delete_undef_values({
    '--interface'      => $intf_name,
    '--url'            => $health_url,
    '--raw-command'    => $_check_command,
    '--interval'       => $interval,    # check interval
    '--window'         => $window,      # history size
    '--threshold'      => $threshold,   # successes required to be healthy
    '--timeout'        => $timeout,     # request timeout
    '--email'          => $alert_email,
    '--nagios-server'  => $nagios_server,
    '--nrdp-folder'    => $nrdp_folder,
    '--nagios-service' => $real_nagios_service,
  }), ' ')
  $daemon_args_string = join($daemon_args, ' ')

  daemontools::supervise { "anycast_health_enforcer_${daemon_name}":
    ensure         => $ensure,
    manage_service => true,
    daemon_dir     => '/var/lib/service',
    wd             => '/tmp',
    syslog         => 'local3',
    predaemon      => [
      "source ${virtenv_dir}/bin/activate",
    ],
    daemon         => "${app_dir}/${script_name} ${daemon_args_string} $connect_flag $noop_flag 2>&1",
  }
}
