# == Define Resource: twitch_anycast::instance
#
# This define will setup a configuration instance for anycast.  The
# following actions are taken:
# * Installs and configures bird internet routing daemon from bird module
# * Creates loopback devices to support the interface
# * Optionally creates instance of anycast health enforcer
#
# == Parameters
# [*address*]
#   Anycast IP address to be configured
#
# [*routed_subnet*]
#   Configure routed subnet to be configured in Bird.  Type: String
#
# [*routed_subnet_service*]
#   Name of service being routed.  Name of static route in Bird.  Type: String
#
# [*interface*]
#   Name of interface to use for anycast.  Type: String
#   Default: 'lo'
#
# [*health_enforcer*]
#   Enable or disable configuring of anycast health enforcer.  Type: Bool
#   Default: false
#
# [*health_check_command*]
#   Command to execute to check for service health.  Type: String
#
# [*health_url*]
#   URL to use by anycast health enforcer.  Type: String
#
# ex Calling the define
# twitch_anycast::instance { "my_anycast_svc":
#   address              => "192.168.1.100",
#   health_enforcer      => true,
#   health_check_command => "/bin/true",
#   health_url           => "http://localhost:1234/health",
# }
#
# ex Using Hiera
# twitch_anycast::instance:
#   my_anycast_svc:
#     address: '192.168.1.100'
#     health_enforcer: true
#     health_check_command: "/bin/true"
#     health_url: "http://localhost:1234/health"
#
define twitch_anycast::instance (
  $address,
  $routed_subnet         = undef,
  $routed_subnet_service = undef,
  $interface             = 'lo',
  $health_enforcer       = false,
  $health_check_command  = undef,
  $health_url            = undef,
  $connect_only          = false,
) {

  validate_bool($health_enforcer)

  if (is_ip_address($address) == false) {
    fail('The anycast ip address is not valid')
  }

  include bird

  # This if-block is used to handle new OpenVPN infra using anycast/bird
  # to advertise routes
  if $routed_subnet != undef {
    bird::daemon { $name:
      routed_subnet         => $routed_subnet,
      routed_subnet_service => $routed_subnet_service,
    }
  }
  else {
    bird::daemon { $name: }
  }

  bird::loopback { "${interface}:${name}":
    ipaddress => $address
  }

  if $health_enforcer {
    twitch_anycast::health_enforcer::daemon { "${interface}:${name}":
      health_url    => $health_url,
      check_command => $health_check_command,
      connect_only  => $connect_only,
    }
  }
}
