# Create an enforcer for a service.
# The name passed in here should match the interface. If you provide a community
# then the interface will be created using bird. If more than one address is
# provided then the interfaces take the names lo:${name}1, lo:${name}2, etc.
#
define twitch_anycast_health_enforcer::service (
  $addresses,
  $noop         = false,
  $ensure       = 'present',
  $interval     = '30s',
  $rise         = undef, # ADDED in 2.0.0
  $fall         = undef, # ADDED in 2.0.0
  $window       = 4,     # REMOVED in 2.0.0
  $threshold    = 3,     # REMOVED in 2.0.0
  $flap_limit   = 6,     # REMOVED in 2.0.0
  $flap_time    = '5m',  # REMOVED in 2.0.0
  $flap_warn    = true,  # REMOVED in 2.0.0
  $healthy      = '',
  $community    = undef,
  $checks       = undef,
  $service_port = undef,
) {
  include twitch_anycast_health_enforcer
  validate_bool($noop,$flap_warn)
  validate_integer($window)
  validate_integer($threshold)
  validate_integer($flap_limit)
  validate_re($ensure,'^(present|absent)$')
  validate_re($healthy,'^(any|all)$|^$', 'healthy can be only "any", "all" or empty ""')
  validate_string($interval,$flap_time)
  validate_array($addresses)

  if versioncmp($twitch_anycast_health_enforcer::version, '2.0.0') >= 0 {
    $new_version = true

    # to handle translation from old paramters to new
    # if we have window=4, threshold=2 this translates to rise=2,fall=2
    $_tmp = $window - $threshold
    $_rise = $rise ? {
      undef   => $_tmp,
      default => $rise,
    }
    $_fall = $fall ? {
      undef   => $_tmp,
      default => $fall,
    }
    validate_integer($_rise)
    validate_integer($_fall)
    if $_fall != $fall or $_rise != $rise {
      notify { "${name}: Rise and Fall parameter not given for new AHE version >= 2.0.0, PLEASE FIX MANIFEST using r:${_rise} and f:${_fall}": }
    }
  }

  $ipnetmasks = suffix($addresses,'/32')                           # ["1.1.1.1/32", "2.2.2.2/32"]
  if size($addresses) > 1 {
    $interfaces = prefix(range(1,size($addresses)),$name)          # ["usher1", "usher2"]
    $interaddrs = hash(zip($ipnetmasks,prefix($interfaces,'lo:'))) # {"1.1.1.1/32" => "lo:usher1", "2.2.2.2/32" => "lo:usher2"}
  } else {
    $interfaces = [$name]                                          # ["usher"]
    $interaddrs = hash([$ipnetmasks[0],"lo:${name}"])              # {"1.1.1.1/32" => "lo:usher"}
  }

  # Concat is used to keep services and their checks in the same file.
  # They may exist in different files; this is for humans.
  concat { "anycast-service-${name}":
    ensure  => $ensure,
    path    => "${twitch_anycast_health_enforcer::service_dir}/${name}.conf",
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    notify  => Class['twitch_anycast_health_enforcer::ahe_service'],
    require => Class['Twitch_anycast_health_enforcer::Config'],
  }

  if $ensure == 'present' {
    concat::fragment { "anycast-service-${name}":
      target  => "anycast-service-${name}",
      order   => '10',
      content => template('twitch_anycast_health_enforcer/anycast-service.erb'),
    }
  }

  if $community != undef {
    # Setup a bird service for each IP address provided.
    validate_integer($community)
    twitch_anycast_health_enforcer::bird { $interfaces:
      addresses    => hash(zip($interfaces,$addresses)),
      community    => $community,
      service_port => $service_port,
    }
  }

  if !empty($checks) {
    validate_hash($checks)
    create_resources(twitch_anycast_health_enforcer::servicecheck, $checks, {service => $name})
  }

  # This cleanup can go away once all core systems hosts are done using old AHE.
  exec { "/usr/local/bin/remove_service anycast_health_enforcer_lo_${name}":
    onlyif => "test -h /etc/service/anycast_health_enforcer_lo_${name}",
  }
}
