# Extra wrapper for cloudwatchlogs. Makes things easier for you!

class twitch_cloudwatchlogs (
  $ensure                        = 'present',
  $use_sandstorm_for_access_keys = true,
  $sandstorm_pfx                 = $twitch_cloudwatchlogs::params::sandstorm_pfx,
  $cloudwatch_access_key_path    = $twitch_cloudwatchlogs::params::cloudwatch_access_key_path,
) inherits twitch_cloudwatchlogs::params {
  include twitch_python::pip::awscli
  require wget

  class { '::cloudwatchlogs':
    ensure  => $ensure,
    region  => 'us-west-2',
    require => Class['twitch_python::pip::awscli', 'wget'],
  }

  if str2bool($::systemd) {
    twitch_systemd::unit_file { 'awslogs.service':
      ensure  => $ensure,
      content => template('twitch_cloudwatchlogs/awslogs.service.erb'),
      before  => Class['cloudwatchlogs'],
    }
  }

  file { '/etc/awslogs/aws.conf':
    ensure  => $ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0400',
    content => template('twitch_cloudwatchlogs/aws.conf.erb'),
    require => Class['cloudwatchlogs'],
  }

  if ($use_sandstorm_for_access_keys) and ($ensure == 'present') {
    $sandstorm_ensure = 'present'
  } else {
    $sandstorm_ensure = 'absent'
  }

  twitch_sandstorm_agent::template { 'cloudwatch_access_key_path':
    ensure          => $sandstorm_ensure,
    destination     => $cloudwatch_access_key_path,
    owner           => 'root',
    group           => 'root',
    mode            => '0400',
    restart_command => 'systemctl restart awslogs',
    contents        => join([
      "AWS_ACCESS_KEY_ID={{ key \"${sandstorm_pfx}/aws_access_key_id\" }}",
      "AWS_SECRET_ACCESS_KEY={{ key \"${sandstorm_pfx}/aws_secret_access_key\" }}",
    ], "\n")
  }

  # This runs every minute.
  cron::job { 'awslogs':
    ensure      => $ensure,
    command     => '/var/awslogs/bin/awslogs-nanny.sh > /dev/null 2>&1',
    environment => ['PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin'],
    require     => Class['cloudwatchlogs'],
    user        => 'root',
  }
  # This deletes the cloudwatchlogs-python-script-made cron.
  # Hourly rotation is moved into logortate::rule above.
  cron::job { 'awslogs_log_rotate':
    ensure => 'absent',
    user   => 'root',
  }
}
