from select import select
import subprocess
import threading

import diamond.collector

METRIC_NAME = "elapsed"

def executemany(l, c):
    """
    Open all commands in l with popen
    select over stdouts until all finish
    return results via callback in c as they complete
    l = [(opaque, [command, list]),]
    c = callback(opaque, raw_text)
    """

    results = {}
    for k, cmd in l:
        p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        results[p.stdout] = (p, k, '')

    while results:
        rlist, _, _ = select([p for p in results.keys()], [], [], 10)
        for r in rlist:
            s = r.read(1)
            p, k, result = results[r]
            if s:
                results[r] = (p, k, result + s)
            else:
                c(k, result)
                del results[r]


class DNSProbeCollector(diamond.collector.Collector):
    def get_default_config(self):
        """
        Returns the default collector settings
        """
        config = super(DNSProbeCollector, self).get_default_config()
        config.update({
            'path': 'dnsprobe',
        })
        return config

    def collect(self):
        """
        Make many digs in parallel and use its output to report how long
        DNS is taking to respond
        """
        commands = []

        # for each probe configured under the probes subsection,
        for probe in self.config['probes']:
            probe_config = self.config['probes'][probe]

            for lookup_key in [l for l in probe_config if l.startswith('lookup')]:
                lookup = probe_config[lookup_key]
                self.log.debug("DNS Probe: %s -> %s"%(probe, lookup))
                # make a metric named after the dns_server.lookup_target.metric_name
                # something like 10_254_0_3.google_com.elapsed
                metric = "%s.%s.%s"%(probe.replace('.', '_'), lookup.replace('.', '_'), METRIC_NAME)
                # and make a dig command like dig @10.254.0.3 google.com
                # and add that to the commands list with the metric as the opaque data for executemany
                commands.append((metric, ('dig', '@'+probe, lookup)))

        def callback(metric, text):
            """
            Find the length of time taken to dig the result and publish it
            """
            published = False
            for l in text.split('\n'):
                if l.startswith(';; Query time:'):
                    published = True
                    self.publish(metric, int(l.split()[3]))
            if not published:
                self.log.error("Unable to find query time in output from %s -> %s"%(probe, lookup))

        # let's do this!
        executemany(commands, callback)
