# coding=utf-8
"""
This is a Twitch-specific complete re-write of https://github.com/python-diamond/Diamond/blob/master/src/diamond/handler/cloudwatch.py, adapted to work with bare metal hosts.
The changes consist of:
     - supporting bare metal metric dimensions.
     - supporting metric rollups
     - metric collection via EMF (depends on local agent by default)
It also removes the concept of rules and sends _all_ metrics from all collectors to Cloudwatch.
Copied on 2018/10/16.
Copied from https://git.xarth.tv/dumbo/config-packages/blob/master/dumbo-monitoring-config-base/fpm_source/usr/local/dumbo/etc/diamond/handlers/DumboCloudwatchHandler.py on 2020/08/11.
--------------

Output the collected values to AWS CloudWatch
Automatically adds the InstanceId Dimension
#### Configuration
Enable this handler
 * handlers = twitch_cloudwatch.cloudwatchHandler
Example Config:
[[twitchCloudwatchHandler]]
environment = 'development'
pop = 'sjc02'
role = 'pop-master'
"""

import json
import logging
import os
import socket

import diamond.handler
from diamond.handler.Handler import Handler

VERSION = '1.0.0'
NAMESPACE = "Diamond"
ROLLUPS = [['environment', 'role'], ['environment', 'pop', 'role']]
LOG_GROUP = '/vidcs/diamond/emf'
SERVICE_DIMENSIONS = [{'Name': 'Service', 'Value': 'Diamond'}]

EMF_PROTO = 'udp'
EMF_HOST = '127.0.0.1'
EMF_PORT = 25888

class TwitchCloudwatchHandler(Handler):
    """
      Implements the abstract Handler class
      Sending data to a AWS CloudWatch
    """

    def __init__(self, config=None):
        """
          Create a new instance of twitchCloudwatchHandler class
        """

        # Initialize Handler
        Handler.__init__(self, config)
        self.enabled = False
        self.log.info('cloudwatch handler: initializing handler..')

        # Twitch specific: Error out if pop or role are not present
        for config_entry in ['pop', 'role', 'environment']:
            if config_entry not in self.config:
                self.log.error("cloudwatch handler: %s not specified, please add it to the config", config_entry)
                return

        # Initialize Data
        self.rollup_dimensions = []
        if 'hostname' not in self.config:
            self.config['hostname'] = socket.gethostname()
        rollups = ROLLUPS + [["environment", "pop", "role", "hostname"]]

        for rollup in rollups:
            d = []
            for dimension_name in rollup:
                d.append(Dimension(dimension_name, self.config[dimension_name]))
            self.rollup_dimensions.append(d)

        self.client = Session(config=self.config)
        self.queue = Queue()
        self.enabled = True
        self.log.info('cloudwatch handler: finished initializing handler..')

    def __del__(self):
        """
          Destroy instance of the twitchCloudWatchHandler class
        """
        try:
            self.log.info('cloudwatch handler: destroying handler instance')
            self.client = None
            self.queue = None
            if self.lock.locked(): self.lock.release()
        except AttributeError:
            pass


    def process(self, metricdata):
        """
          Process a metric and send it to CloudWatch
        """
        if not self.enabled: return

        try:
            self.queue.process(metricdata)
        except Queue.QueueFull:
            self.log.debug('cloudwatch handler: flushing metric data')
            self.client.send_message(metrics=self.queue, dimensions=self.rollup_dimensions)
            self.queue.process(metricdata)


class Session(object):
    def __init__(self, **kwargs):
        self.config = kwargs.get('config')
        self.namespace = kwargs.get('namespace', NAMESPACE)
        self.logGroupName = kwargs.get('logGroupName', LOG_GROUP)
        self.logStreamName = kwargs.get('logStreamName', '.'.join([self.config['hostname'], self.config['pop']]))
        self.log = logging.getLogger('diamond')
        self.log.propagate = True

    def send_message(self, **kwargs):
        metrics = kwargs.get('metrics', list())
        dimensions = kwargs.get('dimensions', list())
        ctx_properties = kwargs.get('properties', dict())
        ctx_metrics = list()
        ctx_dimensions = list()
 
        while metrics:
            m = metrics.pop(0)
            ctx_properties.update({ m['Name']: m.pop('Value') })
            ctx_metrics.append(m)
            ts = m.ts
 
        for rollup in dimensions:
            # [ [foo], [foo, bar] ]
            for d in rollup:
                ctx_properties.update({ d['Name']: d['Value'] })
            ctx_dimensions.append([ i['Name'] for i in rollup ])

        ctx = MetricContext(
             logGroupName=self.logGroupName,
             logStreamName=self.logStreamName,
             namespace=self.namespace,
             dimensions=ctx_dimensions,
             metrics=ctx_metrics,
             properties=ctx_properties,
             ts=ts,
             )

        self.send(ctx)

    def send(self, ctx, proto=EMF_PROTO, host=EMF_HOST, port=EMF_PORT):
        self.log.debug("cloudwatch handler: sending constructed EMF via %s", proto)
        if proto == 'udp':
            message = json.dumps(ctx) + "\n"
            sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            sock.sendto(message.encode('UTF-8'), (host, port))
            sock.close()
        else:
            print(json.dumps(ctx, indent=2))


class Metric(dict):
    def __init__(self, metricdata, unit=None):
        self.ts = metricdata.timestamp * 1000
        self['Name'] = str(metricdata.getCollectorPath()) + "." + str(metricdata.getMetricPath())
        self['Value'] = metricdata.value
        if unit: self['Unit'] = unit

    def __eq__(self, b):
        return self['Name'] == b['Name']

    def __ne__(self, b):
        return not self.__eq__(b)


class Dimension(dict):
    def __init__(self, name, value):
        self['Name'] = name
        self['Value'] = value


class Queue(list):
    def __init__(self, **kwargs):
        self.maxlen = kwargs.get('maxlen', 100)
        self.maxage = kwargs.get('maxage', 0)
        self.last_sent = 0
        self.log = logging.getLogger('diamond')
        self.log.propagate = True

    class QueueFull(Exception): pass

    def process(self, metricdata):
        metric = Metric(metricdata)
        if len(self) > 0 and ( metric in self or len(self) >= self.maxlen or (metric.ts - self.last_sent) > self.maxage ):
            if metric in self:
                self.log.debug("cloudwatch handler: already have metric %s in queue, flushing %s metrics.", metric['Name'], len(self))
            if len(self) >= self.maxlen:
                self.log.debug("cloudwatch handler: queue at max length (%s)", len(self))
            if (metric.ts - self.last_sent) > self.maxage:
                self.log.debug("cloudwatch handler: queue older than max age: %s", self.maxage)
            self.last_sent = metric.ts
            raise Queue.QueueFull
        else:
            self.log.debug("cloudwatch handler: appending metric %s", metric['Name'])
            self.append(metric)


class MetricContext(dict):
    def __init__(self, **kwargs):
        properties = kwargs.get('properties', dict())
        ts = kwargs.get('ts')
        self.update(
            {
              '_aws': {
                'Timestamp': ts,
                'LogGroupName': kwargs.get('logGroupName'),
                'LogStreamName': kwargs.get('logStreamName'),
                'CloudWatchMetrics': [
                  {
                    'Namespace': kwargs.get('namespace'),
                    'Dimensions': kwargs.get('dimensions', list()),
                    'Metrics': kwargs.get('metrics', list()),
                  }
                ]
              },
            'handler_version': VERSION,
            }
        )
        self.update(properties)

