# coding=utf-8

"""
VarnishCollector grabs stats from Varnish and submits them the Graphite

#### Dependencies

 * /usr/bin/varnishstat

"""

import diamond.collector
import re
import subprocess
from diamond.collector import str_to_bool
import os

class VarnishCollector(diamond.collector.Collector):

    _RE = re.compile("^(?P<stat>[\w_.,]*)\s+(?P<psa>\d*)\s+"
                     "(?P<psan>[\d.]*)\s+(?P<desc>.*)$", re.M)

    _KEYS_v4 = re.compile('^(MAIN|MGT|LCK)\.')

    def get_default_config_help(self):
        config_help = super(VarnishCollector, self).get_default_config_help()
        config_help.update({
            'bin':          'The path to the varnishstat binary',
            'instance':     'instance name of the running varnishd',
            'use_sudo':     'Use sudo?',
            'sudo_cmd':     'Path to sudo',
        })
        return config_help

    def get_default_config(self):
        """
        Returns the default collector settings
        """
        config = super(VarnishCollector, self).get_default_config()
        config.update({
            'path':             'varnish',
            'bin':              '/usr/bin/varnishstat',
            'instance':         '/var/lib/varnish/' + os.uname()[1],
            'use_sudo':         False,
            'sudo_cmd':         '/usr/bin/sudo',
        })
        return config

    def collect(self):
        data = {}
        output = self.poll()

        matches = self._RE.findall(output)
        # No matches at all, bail out
        if not matches:
            return

        # Default to a whitelist because there are just too many varnish 4 metrics.
        # Provide your own metrics_whitelist to bypass this default.
        if not self.config.get('metrics_whitelist', None):
            whitelist = self._KEYS_v4.match
        else:
            # Avoid running the regex twice; diamond will do it.
            whitelist = None

        for line in matches:
            if not whitelist or whitelist(line[0]):
                data[line[0]] = line[1]

        for key in data:
            self.log.debug('metric name: {} value: {}'.format(key, int(data[key])))
            self.publish(key, int(data[key]))

    def poll(self):
        try:
            command = [self.config['bin'], '-1', '-n', self.config['instance']]

            if str_to_bool(self.config['use_sudo']):
                command.insert(0, self.config['sudo_cmd'])

            output = subprocess.Popen(command,
                                      stdout=subprocess.PIPE).communicate()[0]
        except OSError:
            output = ""

        return output
