# Installs OpenDKIM Service and Package.
#
class twitch_dkim (
  $dkim_keys,
  $package   = $twitch_dkim::params::package,
  $service   = $twitch_dkim::params::package,
  $socket    = $twitch_dkim::params::socket,
  $user      = $twitch_dkim::params::package,
  $group     = pick($user, $twitch_dkim::params::user),
) inherits twitch_dkim::params {

    include twitch_vars

    $acl_network = concat($::twitch_vars::twitch_networks, '127.0.0.0/8')

    package { $package:
      ensure => present
    }
    validate_hash($dkim_keys)
    if str2bool($::systemd) {
      # render_dkim_key calls out systemd in the sandstorm reloads
      create_resources(twitch_dkim::render_dkim_key, $dkim_keys)
      $notify  = Twitch_systemd::Service[$service]
      $options = "-b s -x /etc/${package}.conf"
      twitch_systemd::service { $service:
        command => '/usr/sbin/opendkim',
        options => "\$DAEMON_OPTS -p \$SOCKET -P /run/twitch_${service}/pid",
        envfile => ["/etc/default/${service}"],
        pidfile => "/run/twitch_${service}/pid",
        user    => $user,
        group   => $group,
        timeout => '10s',
        after   => [
          'network.target',
          'nss-lookup.target',
        ],
        restart => 'always',
      }

    } else {
      $options = '-b s'
      $notify  = Service[$service]
      service { $service:
        ensure    => running,
        enable    => true,
        hasstatus => false,
        require   => Package[$package],
      }
    }
    twitch_servicecheck::passive { 'opendkim-running':
      command => "check_procs -C opendkim -a opendkim -u ${user} -c 1:1",
    }
    file { '/etc/dkim':
      ensure  => directory,
      source  => "puppet:///modules/${module_name}/dkim",
      owner   => $user,
      group   => $group,
      recurse => true,
      require => Package[$package],
      mode    => '0600',
    }

    file { "/etc/default/${service}":
      ensure  => file,
      content => "SOCKET=\"${socket}\"\nDAEMON_OPTS=\"${options}\"\n",
      owner   => 'root',
      group   => 'root',
      notify  => Service[$service],
    }

    $install_files = {
      "/etc/${package}.conf"   => { source => "puppet:///modules/${module_name}/dkim-filter.conf" },
      '/etc/dkim-acl.conf'     => { content => template("${module_name}/dkim-acl.conf.erb") },
      '/etc/dkim-keys.conf'    => { content => template("${module_name}/dkim-keys.conf.erb") },
      '/etc/dkim-signing.conf' => { content => template("${module_name}/dkim-signing.conf.erb") },
    }

    $defaults = {
      'ensure' => present,
      'owner'  => $user,
      'group'  => $group,
      require  => Package[$package],
      notify   => $notify,
    }

    create_resources(file, $install_files, $defaults)
}
