# Grabs a Sandstorm secret that stores a DKIM private and public key.
# Usage:
# 1.) Add secret to "vidcs/dkim/production/SELECTOR.key" as the private key
# 2.) Add secret to "vidcs/dkim/production/SELECTOR.pub: as the public key
# 3.) Ensure your SMTP boxes are granted access to the role serving these keys
# 4.) Include in Hiera:
#  twitch_dkim::dkim_keys:
#      SELECTOR: email-address@here
#
# For more information, see https://wiki.twitch.com/display/SYS/DKIM+runbook
define twitch_dkim::render_dkim_key (
  $email,
  ){
  # $name is the key in this case
  include twitch_dkim
  # Validate we were passed a hash of selector->email
  if $email !~ /@/ {
    fail ("${email} doesn't seem to look like a valid email to me...")
  }
  # split the email and extract the domain
  $email_split = split($email, '@')
  $emaildomain = $email_split[1]
  # Directory for dkim layout
  ensure_resource("file", "/etc/dkim/${emaildomain}",{
    ensure  => 'directory',
    owner   => $twitch_dkim::user,
    group   => $twitch_dkim::group,
    mode    => '0644',
    require => Package[$twitch_dkim::package],
  })
  # DKIM private key
  twitch_sandstorm_agent::template { "dkim-key-${emaildomain}-${name}.private":
    destination     => "/etc/dkim/${emaildomain}/${name}.private",
    key             => "vidcs/dkim/production/${name}.key",
    restart_command => '/bin/systemctl restart opendkim.service',
    owner           => $twitch_dkim::user,
    group           => $twitch_dkim::group,
    mode            => '0400',
    require         => [Package[$twitch_dkim::package], File["/etc/dkim/${emaildomain}"]],
  }
  # DKIM public key
  twitch_sandstorm_agent::template { "dkim-key-${name}.pub":
    destination     => "/etc/dkim/${emaildomain}/${name}.pub",
    key             => "vidcs/dkim/production/${name}.pub",
    restart_command => '/bin/systemctl restart opendkim.service',
    owner           => $twitch_dkim::user,
    group           => $twitch_dkim::group,
    mode            => '0644',
    require         => [Package[$twitch_dkim::package], File["/etc/dkim/${emaildomain}"]],
  }
}
