# Class for installing and configuring entry-pop-collector
class twitch_entrypop (
  $anycast_ip            = undef,
  $anycast_community     = undef,
  $ensure                = $twitch_entrypop::params::ensure,
  $sandstorm_key_prefix  = $twitch_entrypop::params::sandstorm_key_prefix,
  $num_haproxy_instances = $twitch_entrypop::params::num_haproxy_instances,
) inherits twitch_entrypop::params {

  validate_ip_address($anycast_ip)
  validate_integer($anycast_community)
  validate_re($ensure, [ '^present$', '^absent$' ])
  validate_re($sandstorm_key_prefix, '^\w*[\/\w-]+\w*$')
  validate_integer($num_haproxy_instances)

  $appdir = '/opt/twitch/entry-pop-collector'
  $user = 'entry-pop-collector'

  if $ensure == 'absent' {
    $service_ensure = 'stopped'
    $service_enable = false
  } else {
    $service_ensure = 'running'
    $service_enable = true
  }

  twitch::local_user { $user:
    ensure => $ensure,
    group  => $user,
  }

  twitch_systemd::unit_file { 'entry-pop-collector.service':
    ensure  => $ensure,
    content => template("${module_name}/entry-pop-collector.service.erb"),
    require => Twitch::Local_user[$user],
    notify  => Service['entry-pop-collector'],
  }

  service { 'entry-pop-collector':
    ensure     => $service_ensure,
    enable     => $service_enable,
    hasrestart => true,
    require    => [
      Twitch_ssl_certificate::Certificate[wildcard_poe_live-video_net],
      Class['::systemd'],
    ],
  }

  twitch_anycast_health_enforcer::service { 'entrypop_gbl':
    ensure    => $ensure,
    addresses => [
      $anycast_ip,
    ],
    community => $anycast_community,
    checks    => { '/bin/true' => {'type' => 'cmd'} },
    require   => Service['entry-pop-collector'],
  }

  twitch_svclogs::logfile { 'entry-pop-collector':
    ensure             => $ensure,
    use_default_region => true,
  }

  twitch_ssl_certificate::certificate { 'wildcard_poe_live-video_net':
    ensure               => $ensure,
    cert_filesystem_path => '/etc/haproxy/ssl/wildcard_poe_live-video_net.pem',
    cert_contents        => join([
      "{{ key \"${sandstorm_key_prefix}/cert\" }}",
      "{{ key \"${sandstorm_key_prefix}/intermediate\" }}",
      "{{ key \"${sandstorm_key_prefix}/key\" }}",
    ], "\n"),
    cert_owner           => 'root',
    cert_mode            => '0640',
    cert_group           => 'ssl-cert',
  }

  twitch_haproxy::instance { 'entrypop':
    ensure           => $ensure,
    defaults_options => {
      'option' => [
        'redispatch',
        'httplog',
        'tcpka',
      ],
    },
    global_options => {
      'nbproc' => $num_haproxy_instances,
    }
  }

  twitch_haproxy::frontend { 'entrypop':
    instance_name    => 'entrypop',
    ports            => [443],
    ipaddress        => $anycast_ip,
    bind_options     => ['ssl no-sslv3 crt /etc/haproxy/ssl/'],
    options          => {
      'mode'            => 'http',
      'default_backend' => 'entrypop',
      'option'          => [
        'forwardfor',
      ],
    },
    use_so_reuseport => true,
    nbproc           => $num_haproxy_instances,
    require          => [
        Twitch_haproxy::Instance['entrypop'],
        Twitch_ssl_certificate::Certificate['wildcard_poe_live-video_net'],
        Twitch_anycast_health_enforcer::Service['entrypop_gbl'],
    ],
  }

  twitch_haproxy::backend { 'entrypop':
    instance_name       => 'entrypop',
    default_remote_port => 80,
    options             => {
      'mode'   => 'http',
      'option' => [
        'httpchk HEAD /',
      ],
      'http-request' => [
        'set-header X-Entrypop-Anycast %[dst]',
      ],
    },
    server_options      => 'check inter 5000 maxconn 50000',
    server_list         => [
      'localhost:80',
    ],
    require             => Twitch_haproxy::Instance['entrypop'],
  }

}
