#!/bin/bash
#
# used to update inplace ipset tables
# it will determine if a change is needed to the ipset map
# if a change is needed, a temporary ipset map will be created
# and entries added to it, then the old one will be swapped out with
# the new one
#
# usage:
#  ./ipset_helper.sh -a MAP_NAME file_toreadfrom
#  ./ipset_helper.sh -x MAP_TYPE MAP_NAME file_toreadfrom
#
#    -a         : add to existing SetName
#    -x MAP_TYPE: use swap to swap into new map
#
#    MAP_NAME: name of EXISTING ipset map
#    MAP_TYPE: string of map type args to be passed into ipset
#              ex: 'bitmap:port range 8300-8500'
#    ENTRIES: entries to be added to ipset map

IPSET=$(which ipset)

compare_set() {
  current=($($IPSET list $MAP_NAME | /bin/sed '0,/Members:/d'))
  [ ${#IP_LIST[@]} -ne ${#current[@]} ] && return 1 || return 0
}

test_element() {
  $IPSET test -q $MAP_NAME $1
}

destroy_set() {
  echo Destroying temporary set $TMP_MAP_NAME
  $IPSET destroy $TMP_MAP_NAME
}

save_set() {
  save_dir=/etc/iptables/ipset.d
  save_file=$save_dir/$MAP_NAME
  mkdir -p $save_dir
  echo Saving $MAP_NAME state to $save_file
  $IPSET save $MAP_NAME > $save_file
}

recreate_set() {
  echo Creating temporary set $TMP_MAP_NAME
  $IPSET create $TMP_MAP_NAME $MAP_TYPE -exist
  trap destroy_set SIGTERM EXIT
  echo "${IP_LIST[@]}" | /usr/bin/xargs -r -n 1 $IPSET add $TMP_MAP_NAME -exist || exit $?
  echo Swapping temporary set $TMP_MAP_NAME into $MAP_NAME
  $IPSET swap $TMP_MAP_NAME $MAP_NAME
  _ret=$?
  [ $_ret -eq 0 ] && save_set
  exit $_ret
}

add_to_set() {
  echo "${IP_LIST[@]}" | /usr/bin/xargs -r -n 1 $IPSET add -exist $MAP_NAME || exit $?
}

while getopts "ax:" args; do
  case $args in
    a)
      ADD=1
      echo Adding to existing set
      ;;
    x)
      SWAP=1
      MAP_TYPE=$OPTARG
      echo Swapping in new set
      ;;
    \?)
      echo Unknown option
      exit 1
      ;;
    :)
      echo Option -$OPTARG requires argument
      exit 1
      ;;
  esac
done
shift $((OPTIND-1))
MAP_NAME=$1
IP_FILE=$2

if [ -z "$MAP_NAME" ]; then
  echo No MAP_NAME given >&2
  exit 1
fi

$IPSET list $MAP_NAME >/dev/null || exit 1

if [ -z "$IP_FILE" ]; then
  echo Reading ip list from stdin
fi

IP_LIST=()
# read from stdin the list of IPs, command line args have a maximum
count=0
while read ip; do
  IP_LIST[count]=$ip
  count=$((count+1))
done < ${IP_FILE:-/dev/stdin}

if [ ${ADD:-0} -eq 1 ]; then
  add_to_set
elif [ ${SWAP:-0} -eq 1 ]; then
  TMP_MAP_NAME=${MAP_NAME}_${RANDOM:0:5}
  # compare length of lists
  compare_set || recreate_set
  # if we dont obviously have a different sized list
  # walk each individual entry to ensure match
  for x in ${IP_LIST[@]}; do
    test_element $x || recreate_set
  done
fi
# if we haven't touched anything we should
# save current state
save_set
