# Frontier is the service that routes video traffic between Origin and PRs
class twitch_frontier (
  $consul_tags         = $twitch_frontier::params::consul_tags,
  $env                 = $twitch_frontier::params::env,
  $origin_dc           = $twitch_frontier::params::origin_dc,
  $port                = $twitch_frontier::params::port,
  $role                = $twitch_frontier::params::role,
  $frontier_transition = $twitch_frontier::params::frontier_transition,
) inherits twitch_frontier::params {
  # Validate inputs
  validate_string($env, $origin_dc, $port, $role)
  validate_re($env, '^(production|staging)$', 'env invalid, must be {production,staging}')
  validate_re($role, '^(frontier|video-tier1)(|-canary)$', 'role invalid, must be {video-tier1,video-tier1-canary,frontier,frontier-canary}')

  # Values
  $base_path = '/opt/twitch/frontier/current'
  $cfg_dir = '/etc/frontier'
  $cfg_file = "${cfg_dir}/frontier.json"
  $exec_cmd = "${base_path}/frontier --config ${cfg_file}"
  $internal_cfg_file = "${base_path}/config.c7"
  $runtime_dir = 'frontier'
  $pid_file = "/run/${runtime_dir}/pid"
  $systemd_filename = 'frontier.service'
  $user = 'frontier'
  # Needed for twitch_svclogs unless default_region fact can be set to these mappings
  $bound_region = $origin_dc ? {
    'lhr05' => 'eu-west-2',
    default => $::default_region,
  }
  if $bound_region == undef {
    fail("origin_dc (derived from pop fact) not set to a value that can imply bound_region: ${origin_dc}")
  }

  twitch::local_user { $user:
    group => $user,
  }

  # Set up initial config that is passed into frontier
  file { $cfg_dir:
    ensure  => directory,
    path    => $cfg_dir,
    owner   => $user,
    group   => $user,
    require => Twitch::Local_user[$user],
  }

  $cfg = delete_undef_values({
    'BoundRegion'        => $bound_region,
    'Environment'        => $env,
    'InternalConfigFile' => $internal_cfg_file,
    'OriginDC'           => $origin_dc,
    'PIDFile'            => $pid_file,
    'Port'               => $port,
    'Role'               => $role,
  })

  file { $cfg_file:
    ensure  => file,
    path    => $cfg_file,
    owner   => $user,
    group   => $user,
    content => pretty_json($cfg),
    require => File[$cfg_dir],
  }

  # Set up systemd to run service
  twitch_systemd::unit_file { $systemd_filename:
    ensure  => 'present',
    content => template("${module_name}/frontier.service.erb"),
  }

  service { 'frontier':
    ensure     => 'running',
    enable     => true,
    hasrestart => true,
    restart    => 'systemctl reload frontier',
    subscribe  => [
      File[$cfg_file],
      Twitch_systemd::Unit_file[$systemd_filename],
    ],
  }

  # consul service to discover
  # VIDCS-2715 remove condition post tier1/frontier transition.
  if $frontier_transition {
    consul::service { 'video-hls-replication':
      port        => $port,
      consul_tags => concat($consul_tags, "fqdn=${::fqdn}", 'role-frontier', 'role-t1', $::twitch_environment)
    }
  }

  # Set up logging
  twitch_svclogs::logfile { 'frontier':
    ensure             => 'present',
    use_default_region => false,
    region             => $bound_region,
    log_group_name     => "/${role}/${env}/${origin_dc}",
    datetime_format    => '%Y-%m-%dT%H:%M:%S.%f'
  }
  
  # Set up "Frontier Service Health" Montioring
  twitch_servicecheck::passive { 'frontier_service_health':
    command        => "check_http -I 127.0.0.1 -p ${port} -u /health -e 200",
    interval       => 1, # minutes between checks
    retry          => 2, # times to fail before reporting failure
    retry_interval => 1, # minutes between checks once reporting failure
  }
}
