# Installs gotranscodemaster
# Creates consul tags, systemd unit file
# Triggers courier deploy

class twitch_gotranscodermaster (
  $role                = $twitch_gotranscodermaster::params::role,
  $pop                 = $twitch_gotranscodermaster::params::pop,
  $env                 = $twitch_gotranscodermaster::params::env,
  $enable_json_logging = $twitch_gotranscodermaster::params::enable_json_logging,
  $worker_path         = $twitch_gotranscodermaster::params::worker_path,
  $ensure              = $twitch_gotranscodermaster::params::ensure,
  $additional_env      = $twitch_gotranscodermaster::params::additional_env,
  $additional_path     = $twitch_gotranscodermaster::params::additional_path,
  $config_path         = $twitch_gotranscodermaster::params::config_path,
) inherits twitch_gotranscodermaster::params {

  include twitch_proxy_config

  $dir_ensure = $ensure ? {
    'present' => 'directory',
    default   => 'absent',
  }

  # config directory
  file { $config_path:
    ensure => $dir_ensure,
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
  }

  $_no_proxy = join($::twitch_proxy_config::no_proxy,',')

  $_path = $additional_path ? {
    undef   => 'PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin',
    default => "PATH=${additional_path}:/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin"
  }

  $_default_env_vars = [
    'HTTP_PROXY=http://localhost:12395',
    'HTTPS_PROXY=http://localhost:12395',
    "NO_PROXY=${_no_proxy}",
    'http_proxy=http://localhost:12395',
    'https_proxy=http://localhost:12395',
    "no_proxy=${_no_proxy}",
    $_path,
  ]

  $srv_env_vars = concat($_default_env_vars, $additional_env)

  $user = $::lsbdistcodename ? {
    'xenial' => 'jtv',
    default  => 'transcode',
  }

  $group = $::twitch_role ? {
    'transcode-qs' => 'video',
    default        => undef,
  }

  ::consul::service { 'gotranscoder':
    consul_tags   => [
      $env,
    ],
  }

  $_json_log_ensure = $enable_json_logging ? {
    true    => 'present',
    default => 'absent',
  }
  # configure syslog snippet
  rsyslog::snippet { '10-gotranscoder-workers':
    ensure  => $_json_log_ensure,
    content => template("${module_name}/rsyslog_gotranscoder.erb"),
  }

  $_region_map = $pop ? {
    'lhr05' => 'eu-west-2',
    default => $::default_region,
  }

  twitch_svclogs::logfile { 'gotranscoder':
    ensure          => $_json_log_ensure,
    region          => $_region_map,
    log_group_name  => "/ivs/${pop}/gotranscoder/${env}",
    datetime_format => '%Y-%m-%dT%H:%M:%S.%f',
    batch_size      => 1048576, # 1mb max batch size
    batch_count     => 10000, # this is the new default according to the cw docs
  }

  systemd::service { 'gotranscodermaster@':
    ensure            => $ensure,
    description       => 'Transcoder %i Service',
    after             => 'video-milliner.service',
    wants             => ['video-milliner.service'],
    env_vars          => $srv_env_vars,
    restart           => 'always',
    execstartpre      => '/usr/bin/wget --quiet -O /dev/null http://localhost:9090/debug/health',
    execstart         => "${worker_path}/bin/gotranscodermaster \\
                          -config-file /etc/gotranscodermaster/gotranscodermaster-%i.yaml",
    execstop          => '/bin/kill -s TERM $MAINPID',
    execreload        => '/bin/kill -s HUP $MAINPID',
    working_directory => $worker_path,
    syslog_identifier => 'gotranscodermaster-%i',
    syslog_facility   => 'local3',
    environment_files => ['-/etc/milliner-client-env.sh'],
    killmode          => 'process',
    require           => ::Consul::Service['gotranscoder'],
    user              => $user,
    group             => $group,
  }
}
