# == Define Resource Type: twitch_haproxy::backend
#
# This type will setup a backend service configuration block inside
#  the config file on an haproxy load balancer identified by $instance_name.
#  Each backend service configuration needs one or more load balancer member
#  server (that should be pass in via the $server_list param as an array).
#
# === Requirement/Dependencies:
#
# Currently requires the ripienaar/concat module on the Puppet Forge
#
# === Parameters
#
# [*name*]
#  The namevar of the defined resource type is the backend service's name.
#  This name goes right after the 'backend' statement in haproxy.cfg
#
# [*default_remote_port*]
#  This port number will be appended to each server if a port is not part of 
#   server name.
#
# [*options*]
#  A hash of options that are inserted into the backend service
#   configuration block.  This hash will write key / value pairs
#   to the configuration file.  For single world parameters, provide a
#   value of '' (ex. daemon).  For keys that are not unique (ex. option),
#   supply an array of parameters.
#
# [*server_options*]
#  A string of options that will be appended to every server from $server_list
#
# [*instance_name*]
#   The name of to the haproxy instance for which the config will be updated.
#   Allows for multiple haproxy instances on a single machine.
#
# [*backend_template*]
#   Override of the default template to write backend blocks
#
# [*server_template*]
#   Override of the default template to write server lines with backend blocks.
#
# [*server_content*]
#   Override of the server config entries. If used, no server template is used
#   and the caller should generate the proper config entries.
#
# [*source_ips*]
#   An comma seperated string or array of ip addresses that sets the source
#   address which will be used when connecting to the server.
#
# [*server_list*]
#   An array of strings or hashes (can not combine).
#     Strings: The template will parse the string for a ':'.  If one exists,
#       then that port value will be used.  If there is not a ':', the
#       $default_remote_port will be used for port.  The value will also look
#       for a *space*.  Any content will be appended to the server line for
#       only that server.  This provides the ability to have per machine
#       configuration.
#       ex. ['app1.twitch.tv',
#           'app2.twitch.tv:9999',
#           'app3.twitch.tv:9999 backup',
#           'app4.twitch.tv backup'],]
#       output:
#         server app1.twitch.tv-$name app1.twitch.tv:$default_remote_port $server_options
#         server app2.twitch.tv-$name app2.twitch.tv:9999 $server_options
#         server app3.twitch.tv-$name app3.twitch.tv:9999 $server_options backup
#         server app4.twitch.tv-$name app4.twitch.tv:$default_remote_port $server_options backup
#
#     Hashes:  The template will look for the following keys: 'port','hostname','options'.
#     ex. [{'hostname' => 'app1.twitch.tv'},
#          {'hostname' => 'app2.twitch.tv', 'port' => '9999' },
#          {'hostname' => 'app3.twitch.tv', 'port' => '9999', 'options' => ['backup']},
#          {'hostname' => 'app3.twitch.tv','options' => ['backup']},]
#     output:
#       server app1.twitch.tv-$name app1.twitch.tv:$default_remote_port $server_options
#       server app2.twitch.tv-$name app2.twitch.tv:9999 $server_options
#       server app3.twitch.tv-$name app3.twitch.tv:9999 $server_options backup
#       server app4.twitch.tv-$name app3.twitch.tv:$default_remote_port $server_options backup
#
#
# === Authors
#
# Gary Larizza <gary@puppetlabs.com>
# Changes from puppetlabs-haproxy::listen
# - 'mode' removed as define option, can be handled with other $options
# - added 'config_target' define option, in case multiple haproxy instances are needed
# - changed default values in the options define param
# - added 'server_content' so that custom server entries can be used
# - replaced 'config_target' define option with 'instance_name' to reduce coupling
define twitch_haproxy::backend (
  $default_remote_port,
  $instance_name,
  $options          = {
    'mode'    => 'http'
  },
  $backend_template  = 'twitch_haproxy/haproxy_backend_block.erb',
  $server_template  = 'twitch_haproxy/haproxy_listen_block_servers.erb',
  $server_list      = [],
  $server_options   = '',
  $server_content   = '',
  $source_ips       = []
) {
  include twitch_haproxy

  validate_string($instance_name)
  validate_hash($options)
  validate_array($server_list)
  validate_string($server_options)

  $config_target = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.conf"
  validate_absolute_path($config_target)

  # Template uses: $name, $ipaddress, $ports, $options
  concat::fragment { "${name}_backend_block":
    order   => "40-${name}-00",
    target  => $config_target,
    content => template($backend_template),
  }

  $server_contents = $server_content ? {
    ''      => template($server_template),
    default => $server_content,
  }

  concat::fragment { "${name}_backend_block_servers":
    order   => "40-${name}-01",
    target  => $config_target,
    content => $server_contents,
  }
}
