# == Define Resource Type: twitch_haproxy::consul_backend
#
# This type will setup a backend service configuration block inside
#  the config file on an haproxy load balancer identified by $instance_name.
#  Members of the backend service configurations are defined as one or more
#  descriptions of consul services (that should be pass in via the
#  $consul_services param as an array or a single hash)
#
# === Requirement/Dependencies:
#
# Currently requires the ripienaar/concat module on the Puppet Forge
#
# === Parameters
#
# [*name*]
#  The namevar of the defined resource type is the backend service's name.
#  This name goes right after the 'backend' statement in haproxy.cfg
#
# [*options*]
#  A hash of options that are inserted into the backend service
#   configuration block.  This hash will write key / value pairs
#   to the configuration file.  For single world parameters, provide a
#   value of '' (ex. daemon).  For keys that are not unique (ex. option),
#   supply an array of parameters.
#
# [*server_options*]
#  A string of options that will be appended to every server in the configuration
#
# [*instance_name*]
#   The name of to the haproxy instance for which the config will be updated.
#   Allows for multiple haproxy instances on a single machine.
#
# [*backend_template*]
#   Override of the default template to write backend blocks
#
# [*server_template*]
#   Override of the default template to write server lines with backend blocks.
#
# [*server_content*]
#   Override of the server config entries. If used, no server template is used
#   and the caller should generate the proper config entries.
#
# [*consul_services*]
#   An array of hashes or a single hash. The template will look for the
#   following keys: 'service', 'health', 'options'. The first two
#   arguments will be passed directly to consul-template {{service}} directive.
#   Encode the tag, dc and healthiness requirements in those parameters using
#   the consul-template syntax, by default health default to 'any' thus
#   ignoring the health of a service as reported by consul. Value of options
#   entry will be appended to each service entry returned from consul and is
#   empty by default.
#
# === Authors
#
# Gary Larizza <gary@puppetlabs.com>
# Changes from puppetlabs-haproxy::listen
# - 'mode' removed as define option, can be handled with other $options
# - added 'config_target' define option, in case multiple haproxy instances are needed
# - changed default values in the options define param
# - added 'server_content' so that custom server entries can be used
# - replaced 'config_target' define option with 'instance_name' to reduce coupling
# - reimplemented to work with consul template
define twitch_haproxy::consul_backend (
  $instance_name,
  $options          = {
    'mode'    => 'http'
  },
  $backend_template  = 'twitch_haproxy/haproxy_backend_block.erb',
  $server_template  = 'twitch_haproxy/haproxy_consul_listen_block_servers.erb',
  $consul_services  = [],
  $server_options   = '',
  $server_content   = '',
) {
  include twitch_haproxy

  validate_string($instance_name)
  validate_hash($options)
  validate_string($server_options)

  if (!getparam(Twitch_haproxy::Instance[$instance_name], 'with_consul_template')) {
    fail("Cannot define dynamic backend '${title}'. Haproxy instance '${instance_name}' is not configured with consul template enabled.")
  }

  $config_target = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.dynamic.conf.ctmpl"
  validate_absolute_path($config_target)

  # Template uses: $name, $ipaddress, $ports, $options
  concat::fragment { "${name}_backend_block":
    order   => "40-${name}-00",
    target  => $config_target,
    content => template($backend_template),
  }

  # Default template uses: $name, $consul_services, $server_options
  $server_contents = $server_content ? {
    ''      => template($server_template),
    default => $server_content,
  }

  concat::fragment { "${name}_backend_block_servers":
    order   => "40-${name}-01",
    target  => $config_target,
    content => $server_contents,
  }
}
