# == Define Resource Type: twitch_haproxy::consul_listen
#
# This type will setup a listening service configuration block inside
#  the config file on an haproxy load balancer identified by $instance_name.
#  Members of the listening service configurations are defined as one or more
#  descriptions of consul services (that should be pass in via the
#  $consul_services param as an array or a single hash)
#
# === Requirement/Dependencies:
#
# Currently requires the ripienaar/concat module on the Puppet Forge
#
# === Parameters
#
# [*name*]
#   The namevar of the defined resource type is the listening service's name.
#    This name goes right after the 'listen' statement in haproxy.cfg
#
# [*ports*]
#   Ports on which the proxy will listen for connections on the ip address
#    specified in the ipaddress parameter. Accepts either a single
#    comma-separated string or an array of strings which may be ports or
#    hyphenated port ranges.
#
# [*ipaddress*]
#   The ip address the proxy binds to. Empty addresses, '*', and '0.0.0.0'
#    mean that the proxy listens to all valid addresses on the system.
#
# [*options*]
#   A hash of options that are inserted into the listening service
#    configuration block.  This hash will write key / value pairs
#    to the configuration file.  For single world parameters, provide a
#    value of '' (ex. daemon).  For keys that are not unique (ex. option),
#    supply an array of parameters.
#
# [*bind_options*]
#   An array of options that will append to end of bind lines
#
# [*server_options*]
#   A string of options that will be appended to every server in the configuration
#
# [*instance_name*]
#   The name of to the haproxy instance for which the config will be updated.
#   Allows for multiple haproxy instances on a single machine.
#
# [*listen_template*]
#   Override of the default template to write listen blocks
#
# [*server_template*]
#   Override of the default template to write consul template with listen blocks.
#
# [*server_content*]
#   Override of the server config entries. If used, no server template is used
#   and the caller should generate the proper config entries to be consumed by
#   consul template
#
# [*consul_services*]
#   An array of hashes or a single hash. The template will look for the
#   following keys: 'service', 'health', 'options'. The first two
#   arguments will be passed directly to consul-template {{service}} directive.
#   Encode the tag, dc and healthiness requirements in those parameters using
#   the consul-template syntax, by default health defaults to 'any' thus
#   ignoring the health of a service as reported by consul. Value of options
#   entry will be appended to each service entry returned from consul and is
#   empty by default.
#
# [*use_so_reuseport*]
#   Configure haproxy to take advantage of newer kernel option so_reuseport. This new
#   socket option allows multiple sockets on the same host to bind to the same port,
#   and is intended to improve performance. This also properly balance requests across
#   all haproxy processes and cpus. Also set nbproc if this is set to true.
#   Defaults to false.
#
# [*nbproc*]
#   Set this to global_options['nbproc'] to help utilize so_reuseport. Defaults to 1.
#
# === Authors
#
# Gary Larizza <gary@puppetlabs.com>
# Changes from puppetlabs-haproxy::listen
# - 'mode' removed as define option, can be handled with other $options
# - added 'config_target' define option, in case multiple haproxy instances are needed
# - changed default values in the options define param
# - added 'server_content' so that custom server entries can be used
# - replaced 'config_target' define option with 'instance_name' to reduce coupling
# - reimplemented to work with consul template
define twitch_haproxy::consul_listen (
  $instance_name,
  $ports,
  $ipaddress        = [$::ipaddress_primary],
  $options          = {
    'option'  => [
      'forceclose',
      'httplog',
      'dontlog-normal',
      'dontlognull',
      'http-server-close'
    ],
    'balance' => 'roundrobin',
    'mode'    => 'http'
  },
  $bind_options     = [],
  $listen_template  = 'twitch_haproxy/haproxy_listen_block.erb',
  $server_template  = 'twitch_haproxy/haproxy_consul_listen_block_servers.erb',
  $consul_services  = [],
  $server_options   = '',
  $server_content   = '',
  $use_so_reuseport = false,
  $nbproc           = 1,
) {
  include twitch_haproxy

  $reuseport_supported = $::twitch_haproxy::reuseport_supported

  validate_string($instance_name)
  validate_hash($options)
  validate_array($bind_options)
  validate_string($server_options)

  if !is_array($consul_services) and !is_hash($consul_services) {
    fail('consul_services argument is not a hash or array')
  }

  if (!getparam(Twitch_haproxy::Instance[$instance_name], 'with_consul_template')) {
    fail("Cannot define dynamic listener '${title}'. Haproxy instance '${instance_name}' is not configured with consul template enabled.")
  }

  $config_target = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.dynamic.conf.ctmpl"
  validate_absolute_path($config_target)

  # Template uses: $name, $ipaddress, $ports, $options
  concat::fragment { "${name}_listen_block":
    order   => "20-${name}-00",
    target  => $config_target,
    content => template($listen_template),
  }

  # Default template uses: $name, $consul_services, $server_options
  $server_contents = $server_content ? {
    ''      => template($server_template),
    default => $server_content,
  }

  concat::fragment { "${name}_listen_block_servers":
    order   => "20-${name}-01",
    target  => $config_target,
    content => $server_contents,
  }
}
