# == Define Resource Type: twitch_haproxy::frontend
#
# This type will setup a frontend service configuration block inside
#  the config file on an haproxy load balancer identified by $instance_name.
#  Each listening service configuration needs one or more load balancer member
#  server (that should be pass in via the $server_list param as an array).
#
# === Requirement/Dependencies:
#
# Currently requires the ripienaar/concat module on the Puppet Forge
#
# === Parameters
#
# [*name*]
#   The namevar of the defined resource type is the frontend service's name.
#    This name goes right after the 'frontend' statement in haproxy.cfg
#
# [*ports*]
#   Ports on which the proxy will listen for connections on the ip address
#    specified in the ipaddress parameter. Accepts either a single
#    comma-separated string or an array of strings which may be ports or
#    hyphenated port ranges.
#
# [*ipaddress*]
#   The ip address the proxy binds to. Empty addresses, '*', and '0.0.0.0'
#    mean that the proxy listens to all valid addresses on the system.
#
# [*options*]
#   A hash of options that are inserted into the frontend service
#    configuration block.  This hash will write key / value pairs
#    to the configuration file.  For single world parameters, provide a
#    value of '' (ex. daemon).  For keys that are not unique (ex. option),
#    supply an array of parameters.
#
# [*bind_options*]
#   An array of options that will append to end of bind lines
#
# [*server_options*]
#   A string of options that will be appended to every server from $server_list
#
# [*instance_name*]
#   The name of to the haproxy instance for which the config will be updated.
#   Allows for multiple haproxy instances on a single machine.
#
# [*frontend_template*]
#   Override of the default template to write frontend blocks
#
# [*use_so_reuseport*]
#   Configure haproxy to take advantage of newer kernel option so_reuseport. This new
#   socket option allows multiple sockets on the same host to bind to the same port,
#   and is intended to improve performance. This also properly balance requests across
#   all haproxy processes and cpus. Also set nbproc if this is set to true.
#   Defaults to false.
#
# [*nbproc*]
#   Set this to global_options['nbproc'] to help utilize so_reuseport. Defaults to 1.
#
# === Examples
#
#  Exporting the resource for a balancer member:
#
#  twitch_haproxy::frontend { 'puppet00':
#    ipaddress => $::ipaddress_primary,
#    listen_ports     => 18140,
#    options   => {
#      'option'  => [
#        'tcplog',
#        'ssl-hello-chk'
#      ],
#      'balance' => 'roundrobin'
#    },
#  }
#
# === Authors
#
# Gary Larizza <gary@puppetlabs.com>

define twitch_haproxy::frontend (
  $ports,
  $instance_name,
  $ipaddress        = [$::ipaddress_primary],
  $bind_options     = [],
  $options          = {
    'option'  => [
    ],
    'mode'    => 'http'
  },
  $frontend_template  = 'twitch_haproxy/haproxy_frontend_block.erb',
  $use_so_reuseport   = false,
  $nbproc             = 1,
) {
  include twitch_haproxy

  validate_string($instance_name)
  validate_hash($options)
  validate_array($bind_options)

  $config_target = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.conf"
  validate_absolute_path($config_target)

  $reuseport_supported = $::twitch_haproxy::reuseport_supported

  # Template uses: $name, $ipaddress, $ports, $options
  concat::fragment { "${name}_frontend_block":
    order   => "30-${name}-00",
    target  => $config_target,
    content => template($frontend_template),
  }
}
