# == Define: twitch_haproxy::global
#
# Begins the haproxy config for an instance
# Writes the global and default sections
#
# === Parameters
#
# [*global_options*]
#  Hash of options that will create global section of config, Default: <required>
#
# [*defaults_options*]
#  Hash of options that will create defaults section of config, Default: <required>
#
# [*instance_name*]
#   The name of to the haproxy instance for which the config will be updated.
#   Allows for multiple haproxy instances on a single machine.
define twitch_haproxy::global (
  $global_options,
  $defaults_options,
  $instance_name,
  $ensure               = 'present',
  $create_stats_sockets = true,
){
  include twitch_haproxy

  validate_string($instance_name)
  validate_hash($global_options)
  validate_hash($defaults_options)
  validate_re($ensure, [ '^present$', '^absent$' ])

  $config_target = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.conf"
  validate_absolute_path($config_target)

  concat { $config_target:
    ensure  => $ensure,
    owner   => '0',
    group   => '0',
    mode    => '0644',
    require => File[$twitch_haproxy::haproxy_conf_dir],
  }

  $dynamic_config_target = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.dynamic.conf"
  $dynamic_config_target_source = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.dynamic.conf.ctmpl"
  validate_absolute_path($dynamic_config_target)
  validate_absolute_path($dynamic_config_target_source)

  if (getparam(Twitch_haproxy::Instance[$instance_name], 'with_consul_template')) {
    include consul_template

    concat { $dynamic_config_target_source:
      ensure  => $ensure,
      force   => true,
      owner   => '0',
      group   => '0',
      mode    => '0644',
      warn    => "# This file is managed by Puppet and consul-template. DO NOT EDIT.\n",
      require => File[$twitch_haproxy::haproxy_conf_dir],
    }

    if $ensure == 'absent' {
      # Delete the dynamic config file if it exists (and shouldn't) but don't create it.
      file { $dynamic_config_target:
        ensure => 'absent',
      }
    }

    $consul_template_command = str2bool($::systemd) ? {
      true    => "systemctl reload haproxy_${instance_name}",
      default => "/etc/init.d/haproxy_${instance_name} restart",
    }

    # Register our configuration with consul-template
    consul_template::service{ "haproxy_${instance_name}":
      ensure      => $ensure,
      source      => $dynamic_config_target_source,
      destination => $dynamic_config_target,
      command     => $consul_template_command,
    }
  } else {
    # remove dynamic configuration when consul-template is not enabled
    # XXX: hacky solution for dealing with consul-template existing or not such
    # that it won't recreate the file just after we remove it
    file { $dynamic_config_target_source:
      ensure => 'absent',
    }

    Daemontools::Supervise <| title == 'consul-template' |> {
      subscribe +> File[$dynamic_config_target_source],
      before    +> File[$dynamic_config_target],
    }

    file { $dynamic_config_target:
      ensure => 'absent',
    }

  }

  # Simple Header
  if $ensure == 'present' {
    concat::fragment { "00-header-${name}":
      target  => $config_target,
      order   => '01-header',
      content => "# This file managed by Puppet\n",
    }
  }

  # Template uses $global_options, $defaults_options
  if $ensure == 'present' {
    concat::fragment { "haproxy-base-${name}":
      target  => $config_target,
      order   => '10-base',
      content => template('twitch_haproxy/haproxy-base.cfg.erb'),
    }
  }
}
