# == Define: twitch_haproxy::instance
#
# Setup haproxy instance pid/init/conf/service
#
# === Parameters
#
# [*instance_name*]
#  Name of haproxy instance, Default: $name
#
# [*defaults_options*]
#  Hash of options that creates defaults section of config, Default: {}
#
# [*global_options*]
#  Hash of options that creates global section of config, Default: {}
#
# [*haproxy_maxconn*]
#  Haproxy max connections, Default: 32000
#
# [*stats_port*]
#  Listen port for stats listener, Default: 2001
#
# [*stats_listen_address*]
#  Listen address for stats listener, Default: 0.0.0.0
#
# [*enable_url_stats*]
#  true/false - enable stats listener
#
# [*stats_options*]
#  Hash of settings for stats listener, Default: see below
define twitch_haproxy::instance(
  $ensure               = 'present',
  $instance_name        = $name,
  $defaults_options     = {},
  $global_options       = {},
  $haproxy_maxconn      = 32000,
  $stats_port           = 2001,
  $stats_listen_address = '0.0.0.0',
  $enable_url_stats     = true,
  $with_consul_template = false,
  $create_stats_sockets = true,
  $stats_options        = {
    'mode'       => 'http',
    'stats uri'  => '/stats',
    'stats auth' => 'oxygen:atom',
    'option'     => [
      'httplog'
    ],
    'timeout'    => [
      'client 5000',
      'connect 500',
      'server 30000'
    ]
  },
) {
  include twitch_haproxy
  require netcat

  validate_hash($defaults_options)
  validate_hash($global_options)
  validate_hash($stats_options)
  validate_bool($with_consul_template)
  validate_numeric($stats_port)
  validate_bool($enable_url_stats)
  validate_re($ensure, [ '^present$', '^absent$' ])

  $merged_defaults_options = merge($twitch_haproxy::_defaults_options, $defaults_options)
  $temp_global_options     = merge($twitch_haproxy::_global_options, $global_options)
  $server_state_file       = { 'server-state-file' => "haproxy-${instance_name}-global-state" }
  $merged_global_options   = merge($temp_global_options, $server_state_file)
  $dynamic_config_target   = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.dynamic.conf"
  $config_target           = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.conf"
  $restart_command         = str2bool($::systemd) ? {
    true    => "/bin/systemctl reload haproxy_${instance_name}",
    default => undef
  }

  if ($::twitch_haproxy::haproxy_systemd_wrapper) {
    $systemd_exec_start = '/usr/local/sbin/haproxy'
  } else {
    $systemd_exec_start = '/usr/local/sbin/haproxy-systemd-wrapper'
  }

  validate_absolute_path($config_target)
  validate_absolute_path($dynamic_config_target)

  twitch_haproxy::global{ $instance_name:
    ensure               => $ensure,
    instance_name        => $instance_name,
    defaults_options     => $merged_defaults_options,
    global_options       => $merged_global_options,
    create_stats_sockets => $create_stats_sockets,
    require              => Class['twitch_haproxy'],
  }

  if ($enable_url_stats == true) {
    $stats_ports_array = any2array($stats_port)
    if size($stats_ports_array) > 1 and size($stats_ports_array) != $merged_global_options['nbproc'] {
      fail { "haproxy ${instance_name} global 'nbproc' value must match the number of stat ports specified": }
    }

    # for backwards compatibility we don't want to bind stats to a specific process
    # so only create the port to process mapping if given an array of stats ports
    if is_array($stats_port) {
      $port_to_process = hash(flatten(zip($stats_ports_array, range(1, size($stats_ports_array)))))
    } else {
      $port_to_process = {}
    }

    twitch_haproxy::stats { $stats_ports_array:
      ensure          => $ensure,
      instance_name   => $instance_name,
      options         => $stats_options,
      ip              => $stats_listen_address,
      port_to_process => $port_to_process,
    }
  }

  if $ensure == 'present' {
    $service_ensure = 'running'
    $service_enable = true

    exec { "Draining old ${instance_name} managed by init.d":
      command     => "pkill -TTOU -P 1 -u www-data  -f /etc/haproxy/haproxy-${instance_name}.conf && pkill -USR1 -P 1 -u www-data  -f /etc/haproxy/haproxy-${instance_name}.conf",
      onlyif      => "pgrep -P 1 -u www-data  -f /etc/haproxy/haproxy-${instance_name}.conf",
      refreshonly => true,
      require     => Service["haproxy_${instance_name}"],
    }

    exec {"Updated systemd support in haproxy ${instance_name} requires a restart":
      command     => "systemctl restart haproxy_${instance_name}",
      onlyif      => [
        'test ! -x /usr/local/sbin/haproxy-systemd-wrapper',
        'test -x /usr/local/sbin/haproxy',
        "/usr/bin/pgrep -f /usr/local/sbin/haproxy-systemd-wrapper.*/etc/haproxy/haproxy-${instance_name}.conf",
      ],
      refreshonly => true,
      require     => Service["haproxy_${instance_name}"],
    }

    if str2bool($::systemd) {
      twitch_systemd::unit_file { "haproxy_${instance_name}.service":
        ensure  => $ensure,
        content => template("${module_name}/haproxy.service.erb"),
        notify  => [
          Service["haproxy_${instance_name}"],
          Exec["Draining old ${instance_name} managed by init.d"],
          Exec["Updated systemd support in haproxy ${instance_name} requires a restart"],
        ],
        subscribe => Package['haproxy'],
      }
    }
  } else {
    $service_ensure = 'stopped'
    $service_enable = false
  }

  service { "haproxy_${instance_name}":
    ensure     => $service_ensure,
    enable     => $service_enable,
    hasrestart => true,
    restart    => $restart_command,
    hasstatus  => true,
    subscribe  => [
      Package['haproxy'],
      Twitch_haproxy::Global[$instance_name],
      Twitch_haproxy::Stats[$stats_ports_array],
      File["/etc/init.d/haproxy_${instance_name}"],
    ],
    require    => Class['twitch_haproxy'],
  }

  # we need to signal to consul-template to refresh the dynamic config before we try to restart haproxy
  if $ensure == 'present' and $with_consul_template {
    Service <| title == "haproxy_${instance_name}" |> {
      require +> Service['consul-template']
    }
  }

  # Template uses $with_consul_template, $haproxy_maxconn, $instance_name, $config_target, $dynamic_config_target
  file { "/etc/init.d/haproxy_${instance_name}":
    ensure  => $ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    content => template('twitch_haproxy/haproxy-init.erb'),
    require => Package['netcat-openbsd'],
    ##  netcat-openbsd??
  }
}
