# == Define: twitch_haproxy::resolver
#
# Writes the resolvers sections
#
# === Parameters
#
# [*nameservers*]
#   Defines the nameservers to use for dns lookups
#       Default value: hiera('twitch_nameservers')
#
# [*resolve_retries*]
#   Defines the number <nb> of queries to send to resolve a server name before
#   giving up.
#       Default value: 3
#
# [*timeout_retry*]
#   Defines time between two DNS queries, when no response have been received.
#       Default value: 1s
#
# [*accepted_payload_size*]
#   Defines the maximum payload size accepted by HAProxy and announced to all the
#   name servers configured in this resolvers section.
#   taken from: https://cbonte.github.io/haproxy-dconv/2.0/configuration.html#5.3.2-accepted_payload_size
#
#       Default value: undef (haproxy defaults to 512)
#
# [*holds*]
#   Defines various periods during which the last name resolution should be kept
#   based on last resolution
#   taken from: https://cbonte.github.io/haproxy-dconv/1.7/configuration.html#5.3.2
#       Default value: {}
#
# [*instance_name*]
#   The name of to the haproxy instance for which the config will be updated.
#   Allows for multiple haproxy instances on a single machine.
define twitch_haproxy::resolver(
  $instance_name,
  $nameservers           = hiera('twitch_nameservers'),
  $resolve_retries       = 3,
  $timeout_retry         = '1s',
  $accepted_payload_size = undef,
  $holds                 = {},
){
  include twitch_haproxy

  validate_string($instance_name, $timeout_retry)
  validate_hash($holds)
  validate_array($nameservers)
  validate_integer($resolve_retries)
  if size($nameservers) == 0 {
    fail("\$nameservers cannot be empty.")
  }

  $config_target = "${twitch_haproxy::haproxy_conf_dir}/haproxy-${instance_name}.conf"
  validate_absolute_path($config_target)

  # Template uses $options
  concat::fragment { "haproxy-resolver-${name}":
    target  => $config_target,
    order   => '11-resolver',
    content => template('twitch_haproxy/haproxy-resolver.cfg.erb'),
  }
}
