# helper to facilitate conventions on paths for ssl certificates
#
# a certificate can be:
#  1. separate public/private pair (most common)
#  2. PEM file encoded with both private and public info
define twitch_haproxy::ssl_certificate (
  $instance_name        = $name,
  $sandstorm_key_prefix = undef,
  $use_twitch_wildcard  = false,
  $alg_type             = 'rsa',
) {
  require twitch_haproxy
  validate_absolute_path($twitch_haproxy::haproxy_conf_dir)
  if !$use_twitch_wildcard {
    validate_re($sandstorm_key_prefix, '^\w*[\/\w-]+\w*$')
  }

  validate_re($alg_type, '^(rsa|ecdsa|both)$', 'The value for alg_type must be either "rsa", "ecdsa" or "both"')

  $restart_command = str2bool($::systemd) ? {
    true    => "/bin/systemctl try-reload-or-restart haproxy_${instance_name}",
    default => "/etc/init.d/haproxy_${instance_name} restart"
  }

  $rsa_contents = $use_twitch_wildcard ? {
    true    => join([
      '{{ key "syseng/ssl_keys/production/wildcard.twitch.tv" }}',
      '{{ key "syseng/ssl_certs/production/wildcard.twitch.tv" }}',
      '{{ key "syseng/ssl_certs/production/wildcard.twitch.tv.dhparams" }}',
    ], "\n"),
    default => template('twitch_haproxy/sandstorm/pem.rsa.erb'),
  }

  # Purge previoius pem location. New pems will not load if present
  file { "${twitch_haproxy::haproxy_conf_dir}/ssl/${name}.pem":
    ensure => absent,
  }

  if $alg_type =~ /^(rsa|both)$/ {

    twitch_ssl_certificate::certificate { "${name}.rsa":
      cert_filesystem_path => "${twitch_haproxy::haproxy_conf_dir}/ssl/${name}.rsa",
      cert_contents        => $rsa_contents,
      cert_owner           => 'root',
      cert_group           => 'ssl-cert',
      cert_mode            => '0640',
      cert_restart_cmd     => $restart_command,
      require              => File["${twitch_haproxy::haproxy_conf_dir}/ssl/${name}.pem"],
    }

  }

  if $alg_type =~ /^(ecdsa|both)$/ {

    twitch_ssl_certificate::certificate { "${name}.ecdsa":
      cert_filesystem_path => "${twitch_haproxy::haproxy_conf_dir}/ssl/${name}.ecdsa",
      cert_contents        => template('twitch_haproxy/sandstorm/pem.ecdsa.erb'),
      cert_owner           => 'root',
      cert_group           => 'ssl-cert',
      cert_mode            => '0640',
      cert_restart_cmd     => $restart_command,
      require              => File["${twitch_haproxy::haproxy_conf_dir}/ssl/${name}.pem"],
    }

  }

}
