# == Define Resource Type: twitch_haproxy::static_consul_listen
#
# This type will setup a listening service configuration block inside
#  the config file on an haproxy load balancer identified by $instance_name.
#  Each listening service configuration needs one or more load balancer member
#  server (that should be pass in via the $server_list param as an array).
#
# === Requirement/Dependencies:
#
# Currently requires the ripienaar/concat module on the Puppet Forge
#
# === Parameters
#
# [*name*]
#   The namevar of the defined resource type is the listening service's name.
#    This name goes right after the 'listen' statement in haproxy.cfg
#
# [*ports*]
#   Ports on which the proxy will listen for connections on the ip address
#    specified in the ipaddress parameter. Accepts either a single
#    comma-separated string or an array of strings which may be ports or
#    hyphenated port ranges.
#
# [*ipaddress*]
#   The ip address the proxy binds to. Empty addresses, '*', and '0.0.0.0'
#    mean that the proxy listens to all valid addresses on the system.
#
# [*options*]
#   A hash of options that are inserted into the listening service
#    configuration block.  This hash will write key / value pairs
#    to the configuration file.  For single world parameters, provide a
#    value of '' (ex. daemon).  For keys that are not unique (ex. option),
#    supply an array of parameters.
#
# [*bind_options*]
#   An array of options that will append to end of bind lines
#
# [*server_options*]
#   A string of options that will be appended to every server from $server_list
#
# [*instance_name*]
#   The name of to the haproxy instance for which the config will be updated.
#   Allows for multiple haproxy instances on a single machine.
#
# [*listen_template*]
#   Override of the default template to write listen blocks
#
# [*server_template*]
#   Override of the default template to write server blocks
#
# [*server_options*]
#   Server options to include for each server
#
# [*consul_services*]
#   Output from `consul_service_lookup` function
#
# [*prefer_localhost*]
#   When true, non-localhost servers will be considered backups
#

define twitch_haproxy::static_consul_listen (
  $ports,
  $consul_services,
  $instance_name,
  $ipaddress        = [$::ipaddress_primary],
  $options          = {
    'option'  => [
      'forceclose',
      'httplog',
      'dontlog-normal',
      'dontlognull',
      'http-server-close'
    ],
    'balance' => 'roundrobin',
    'mode'    => 'http'
  },
  $bind_options     = [],
  $listen_template  = 'twitch_haproxy/haproxy_listen_block.erb',
  $server_template  = 'twitch_haproxy/haproxy_static_consul_listen_block_servers.erb',
  $server_options   = '',
  $prefer_localhost = false,
) {

  validate_string($instance_name)
  validate_array($consul_services)

  twitch_haproxy::listen { $name:
    instance_name       => $instance_name,
    ports               => $ports,
    ipaddress           => $ipaddress,
    options             => $options,
    bind_options        => $bind_options,
    listen_template     => $listen_template,
    server_options      => $server_options,
    server_content      => template($server_template),
    default_remote_port => -1,
  }
}
