# Intelligest class for installing and confiuring intelligest
class twitch_intelligest(
  $env                    = $twitch_intelligest::params::env,
  $ulimit_n               = $twitch_intelligest::params::ulimit_n,
  $substage               = $twitch_intelligest::params::substage,
  $rtmp_port              = $twitch_intelligest::params::rtmp_port,
  $rtmps_port             = $twitch_intelligest::params::rtmps_port,
  $aws_region             = $twitch_intelligest::params::aws_region,
  $consul_addr            = $twitch_intelligest::params::consul_addr,
  $num_instances          = $twitch_intelligest::params::num_instances,
  $max_connections        = $twitch_intelligest::params::max_connections,
  $is_private_ingest      = $twitch_intelligest::params::is_private_ingest,
  $host_check_interval    = $twitch_intelligest::params::host_check_interval,
  $consul_check_interval  = $twitch_intelligest::params::consul_check_interval,
  $pop_map_check_interval = $twitch_intelligest::params::pop_map_check_interval,
  $platform               = $twitch_intelligest::params::platform,
  $backend_provider       = $twitch_intelligest::params::backend_provider,
) inherits twitch_intelligest::params {

  include twitch_intelligest::certs
  validate_bool($is_private_ingest)

  # private variables, should not be user selectable
  $base_path        = $::twitch_intelligest::params::base_path
  $tls_dir          = $::twitch_intelligest::params::tls_dir
  $config_file      = $::twitch_intelligest::params::config_file
  $user             = $::twitch_intelligest::params::user
  $group            = $::twitch_intelligest::params::group
  $systemd_exec_cmd = "${base_path}/current/intelligest --config ${config_file}"

  case $env {
    'production': {
      $stage = 'production'
      $certs = [
        { 'Cert' => "${tls_dir}/wildcard_twitch_tv.crt", 'Key' => "${tls_dir}/wildcard_twitch_tv.key" },
        { 'Cert' => "${tls_dir}/wildcard_contribute_live-video_net.crt", 'Key' => "${tls_dir}/wildcard_contribute_live-video_net.key" },
        { 'Cert' => "${tls_dir}/wildcard_global-contribute_live-video_net.crt", 'Key' => "${tls_dir}/wildcard_global-contribute_live-video_net.key" },
        { 'Cert' => "${tls_dir}/wildcard_rist_live-video_net.crt", 'Key' => "${tls_dir}/wildcard_rist_live-video_net.key" },
        { 'Cert' => "${tls_dir}/wildcard_webrtc_live-video_net.crt", 'Key' => "${tls_dir}/wildcard_webrtc_live-video_net.key" },
      ]
    }
    default: {
      $stage = 'staging'
      $certs = [
        { 'Cert' => "${tls_dir}/wildcard_twitch_tv.crt", 'Key' => "${tls_dir}/wildcard_twitch_tv.key" },
        { 'Cert' => "${tls_dir}/wildcard_staging_contribute_live-video_net.crt", 'Key' => "${tls_dir}/wildcard_staging_contribute_live-video_net.key" },
        { 'Cert' => "${tls_dir}/wildcard_rist-staging_live-video_net.crt", 'Key' => "${tls_dir}/wildcard_rist-staging_live-video_net.key" },
        { 'Cert' => "${tls_dir}/wildcard_webrtc-staging_live-video_net.crt", 'Key' => "${tls_dir}/wildcard_webrtc-staging_live-video_net.key" },
      ]
    }
  }

  # these are dynamically determined based on environment above
  $addr_rtmp   = "${::ipaddress_primary}:${rtmp_port}"
  $addr_rtmps  = "${::ipaddress_primary}:${rtmps_port}"
  $consul_tags = unique(flatten([$stage, $env]))

  twitch_systemd::unit_file { 'intelligest@.service':
    ensure  => 'present',
    content => template("${module_name}/intelligest@.service.erb"),
  }

  twitch_systemd::unit_file { 'intelligest.service':
    ensure  => 'present',
    content => template("${module_name}/intelligest.service.erb"),
  }

  # need to have local user created and group
  twitch::local_user { $user:
    group  => $group,
    before => [
      File[$tls_dir],
      Class['twitch_intelligest::certs'],
    ]
  }

  file { $base_path:
    ensure => 'directory',
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
  }

  file { $::twitch_intelligest::params::conf_dir:
    ensure => 'directory',
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
  }

  file { $::twitch_intelligest::params::module_dir:
    ensure  => 'directory',
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    recurse => true,   # ensure puppet manages all the files in this directory
    purge   => true,
  }

  file { $tls_dir:
    ensure => 'directory',
    owner  => 'root',
    group  => 'intelligest',
    mode   => '0640',
  }

  $_module_classes = prefix($twitch_intelligest::params::enabled_modules, 'twitch_intelligest::modules::')
  class { $_module_classes: }

  # Create final config hash for YAML file rendering
  $options = delete_undef_values({
    'AWSRegion'            => $aws_region,
    'AddrRTMP'             => $addr_rtmp,
    'AddrRTMPS'            => $addr_rtmps,
    'MaxConns'             => $max_connections*1,
    'BackendProvider'      => $backend_provider,
    'BackendCheckInterval' => $consul_check_interval,
    'ConsulAddr'           => $consul_addr,
    'ConsulCheckInterval'  => $consul_check_interval,
    'HostCheckInterval'    => $host_check_interval,
    'PoP'                  => $::facts['pop'],
    'Stage'                => $stage,
    'Substage'             => $substage,
    'PoPMapCheckInterval'  => $pop_map_check_interval,
    'IsPrivateIngest'      => $is_private_ingest,
    'TLSCerts'             => $certs,
    'ModuleConfigDir'      => $::twitch_intelligest::params::module_dir,
    'Platform'             => $platform,
  })

  # intelligest config file
  file { $config_file:
    ensure  => 'file',
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => template("${module_name}/intelligest.yaml.erb"),
    notify  => Service['intelligest'],
  }

  twitch_svclogs::logfile { 'intelligest':
    ensure             => 'present',
    use_default_region => true,
    log_group_name     => 'intelligest',
    datetime_format    => '%Y-%m-%dT%H:%M:%S',
  }

  # XXX: In order to properly migrate from HAproxy to Intelligest
  #      we need to order intelligest starting after the Haproxy service
  #      is declared 'stopped'
  if defined(Class['twitch::haproxy::tolive']) {
    notify { 'Intelliest: haproxy still defined, ensure we start after its controlled': }
    Service <| title == 'twitch::haproxy::tolive' |> {
      require +> Class['twitch::haproxy::tolive']
    }
  }

  service { 'intelligest':
    ensure  => 'running',
    enable  => true,
    restart => '/bin/systemctl reload intelligest',
    require => [
      Class['::twitch_intelligest::certs'],
      Class['::systemd'],
    ]
  }

}
