# Opendj configuration settings
class twitch_ldap_server::config (
  $ldaps_port      = $twitch_ldap_server::params::ldaps_port,
  $ldap_port       = $twitch_ldap_server::params::ldap_port,
  $admin_port      = $twitch_ldap_server::params::admin_port,
  $base_dn         = $twitch_ldap_server::params::base_dn,
  $admin_dn        = $twitch_ldap_server::params::admin_dn,
  $ss_dm_pass      = "syseng/ldap/${twitch_ldap_server::env}/directory_manager",
  $ss_truststore   = "syseng/ldap/${twitch_ldap_server::env}/truststore",
  $ss_keystore     = "syseng/ldap/${twitch_ldap_server::env}/keystore",
  $setup_pass_file = $twitch_ldap_server::params::setup_pass_file,
  $heap_size       = '24g',
  $new_ratio       = '1',
  $install_dir     = $twitch_ldap_server::params::install_dir,
) inherits twitch_ldap_server::params {

  include ::twitch_ldap_server

  validate_integer([$ldaps_port, $ldap_port, $admin_port])
  validate_string($base_dn, $admin_dn)

  $schema_path = "${install_dir}/db/schema"
  $config_path = "${install_dir}/config"

  twitch_sandstorm_agent::resource { 'opendj_setup':
    key             => $ss_dm_pass,
    cwd             => '/opt/opendj',
    user            => 'root',
    group           => 'root',
    exists_command  => "/usr/bin/test -a ${schema_path} || /usr/bin/test -a ${config_path}/schema",
    create_command  => "/opt/opendj/setup -Q --acceptLicense -O -j ${setup_pass_file} -p ${ldap_port} -Z ${ldaps_port} -h ${::fqdn} -D '${admin_dn}' --adminConnectorPort ${admin_port} -b '${base_dn}' --doNotStart",
    destroy_command => '/bin/true',
    in_sync_command => '/bin/true',
    update_command  => '/bin/true',
    require         => Package['opendj'],
    notify          => Exec['opendj_fix_jks_provider'],
  }

  exec { 'opendj_fix_jks_provider':
    command     => "${install_dir}/bin/dsconfig set-key-manager-provider-prop --provider-name 'Default Key Manager' --set key-store-type:JKS  --no-prompt --offline",
    refreshonly => true,
    require     => Twitch_sandstorm_agent::Resource['opendj_setup'],
  }

  exec { 'fix_upgrade_schema_location':
    creates => $schema_path,
    command => "service opendj stop && mv ${config_path}/schema ${schema_path}",
    require => Twitch_sandstorm_agent::Resource['opendj_setup'],
    notify  => Twitch_systemd::Service['opendj'],
  }

  file { "${config_path}/business-names.txt":
    source  => "puppet:///modules/${module_name}/business-names.txt",
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    require => Twitch_sandstorm_agent::Resource['opendj_setup'],
  }

  file { "${schema_path}/90-openssh.ldif":
    source  => "puppet:///modules/${module_name}/openssh-lpk_sun.schema",
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    require => [ Twitch_sandstorm_agent::Resource['opendj_setup'], Exec['fix_upgrade_schema_location'] ],
  }

  file { "${config_path}/java.properties":
    ensure  => 'file',
    content => template("${module_name}/java.properties.erb"),
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    require => Twitch_sandstorm_agent::Resource['opendj_setup'],
    notify  => Service['opendj'],
  }

  $_keystore_pin_path = "${config_path}/keystore.pin"
  # Place the keystore + pin
  twitch_ssl_certificate::certificate { 'java_keystore':
    cert_filesystem_path     => "${config_path}/keystore",
    cert_contents            => "{{ key \"${ss_keystore}\" }}",
    cert_mode                => '0600',
    cert_ssl_host            => 'localhost',
    cert_ssl_port            => $ldaps_port,
    key_identifier           => 'java_keystore.pin', # mechanical sympathy
    key_filesystem_path      => $_keystore_pin_path,
    key_mode                 => '0600',
    key_contents             => "{{ key \"${ss_keystore}.pin\" }}",
    monitoring_check_fs_cert => false,
    require                  => Twitch_sandstorm_agent::Resource['opendj_setup'],
  }

  # Place the truststore (just the cert) Has the same monitoring check.
  twitch_ssl_certificate::certificate { 'java_truststore':
    cert_filesystem_path     => "${config_path}/truststore",
    cert_contents            => "{{ key \"${ss_truststore}\" }}",
    cert_mode                => '0600',
    cert_ssl_host            => 'localhost',
    cert_ssl_port            => $ldaps_port,
    monitoring_check_fs_cert => false,
    require                  => Twitch_sandstorm_agent::Resource['opendj_setup'],
  }


  file { '/etc/ldap/ldap.conf':
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
    source => "puppet:///modules/${module_name}/ldap.conf",
  }
}
