# Setup required users for opendj
define twitch_ldap_server::user (
  $password_key,
  # Since creating LDAP users is so funky, we're only supporting "admin" users
  # for now. I'm making it a required parameter so that the user is always aware
  # of the fact that setting it to false is not supported.
  $admin,
  $ensure          = 'present',
  $username        = $title,
) {
  include twitch_ldap_server
  include twitch_ldap_server::params

  # TODO: Support non-admin users. We only need this resource for
  # "replicationadmin" at the moment, so there's no point implimenting the
  # logic to switch the LDIF in the create_command script for non-admins.
  if !defined(Twitch_sandstorm_agent::Resource['opendj_setup']) {
    fail('twitch_ldap_server::user requires Twitch_sandstorm_agent::Resource[\'opendj_setup\']')
  }
  $setup_pass_file = $twitch_ldap_server::params::setup_pass_file

  validate_re($ensure, ['^present$', '^absent$'])
  validate_string($password_key)
  validate_bool($admin)
  validate_string($username)


  if $admin {
    $_base_dn = 'cn=Administrators,cn=admin data'
    $_user_dn = "cn=${username},${_base_dn}"
  } else {
    fail("Non-admin LDAP users aren't supported quite yet :(")
  }

  $_script_directory = '/var/lib/sandstorm-agent/scripts'

  # Managing the user always requires use of the LDAP directory manager
  # password, so we wrap these commands in scripts so they can do that. They
  # also sometimes need huge ugly LDIFs, and we don't want to touch that.
  $_create_script = "${_script_directory}/create-opendj_user_${username}.sh"
  file { $_create_script:
    ensure  => $ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0700',
    content => template("${module_name}/user_create.sh.erb"),
  }

  $_delete_script = "${_script_directory}/delete-opendj_user_${username}.sh"
  file { $_delete_script:
    ensure  => 'present',
    owner   => 'root',
    group   => 'root',
    mode    => '0700',
    content => template("${module_name}/user_delete.sh.erb"),
  }

  $_update_script = "${_script_directory}/custom-update-opendj_user_${username}.sh"
  file { $_update_script:
    ensure  => 'present',
    owner   => 'root',
    group   => 'root',
    mode    => '0700',
    content => template("${module_name}/user_update.sh.erb"),
  }


  twitch_sandstorm_agent::resource { "opendj_user_${username}":
    ensure          => $ensure,
    key             => $password_key,
    exists_command  => "/usr/bin/ldapsearch -H ldaps://localhost/ 'cn=${username}' -x -w '' -b '${_base_dn}' | grep numEntries",
    create_command  => "${_create_script} %q",
    destroy_command => $_delete_script,
    in_sync_command => "/usr/bin/ldapwhoami -H ldaps://localhost/ -D '${_user_dn}' -x -w %q",
    update_command  => "${_update_script} %q",
    require         => Service['opendj'],
  }
}
