# Installs and managed LMD service
class twitch_lmd (
  $lmd_version = 'present',
  $installed   = true,
  $running     = true,
  $env = pick($twitch_environment, $::twitch_environment, 'staging'),
) {

  $_lmd_ensure = $installed ? {
    false => 'absent',
    true  => 'present',
  }

  include twitch_lmd::metrics

  package { 'lmd':
    ensure  => $lmd_version,
  }

  file { '/etc/lmd/lmd.ini.ctmpl':
    ensure  => $_lmd_ensure,
    owner   => 'lmd',
    mode    => '0644',
    content => template('twitch_lmd/lmd.ini.ctmpl.erb'),
    require => Package['lmd'],
  }

  file { '/etc/lmd/lmd.ini':
    ensure  => $_lmd_ensure,
    replace => 'no',
    owner   => 'lmd',
    group   => 'lmd',
    mode    => '0644',
    require => Package['lmd'],
  }

  consul_template::service{ 'twitch_lmd':
    ensure          => $_lmd_ensure,
    source          => '/etc/lmd/lmd.ini.ctmpl',
    destination     => '/etc/lmd/lmd.ini',
    command         => 'sudo systemctl restart lmd',
    command_timeout => '2m',
    require         => File['/etc/lmd/lmd.ini.ctmpl'],
  }

  systemd::service { 'lmd':
    ensure            => $_lmd_ensure,
    description       => 'Livestatus Multitool Daemon',
    user              => 'lmd',
    group             => 'lmd',
    restart           => 'on-failure',
    restart_sec       => 1,
    execstart         => '/usr/bin/lmd -v -c /etc/lmd/lmd.ini',
    working_directory => '/usr/local/bin',
    syslog_facility   => 'local3',
    standard_output   => 'journal',
    standard_error    => 'journal',
    require           => Package['lmd'],
    notify            => Service['lmd'],
  }

  service { 'lmd':
      ensure     => $running,
      enable     => $running,
      hasstatus  => true,
      hasrestart => true,
      require    => Systemd::Service['lmd'],
  }

  # Don't mess with this directory if LMD is not installed
  if $installed {
    file { ['/var/nagios/rw', '/var/nagios']:
      ensure => 'directory',
      owner  => 'lmd',
      group  => 'www-data',
    }
  }

  # Monitoring
  twitch_servicecheck::passive { 'check_lmd_status':
    command        => '/usr/local/bin/check-systemd-service -s lmd',
    interval       => 5, # minutes between checks
    retry          => 3, # times to fail before reporting failure
    retry_interval => 2, # minutes between checks once reporting failure
  }
}
