#!/usr/bin/env python

import argparse
import requests
import json
import sys
import re
import socket
import os
consul_endpoint = 'http://localhost:8500/v1'

def do_request(endpoint,hostname,dc):
    req              = {}
    url              = consul_endpoint + endpoint + hostname + '?stale&dc=' + dc
    request          = requests.get(url)
    req['response']  = request
    req['request']   = url
    req['body']      = request.text
    return req

def pretty_print_req(req):
    print('{}\n{}\n{}\n\n{}'.format(
        '-----------START-----------',
        req.method + ' ' + req.url,
        '\n'.join('{}: {}'.format(k, v) for k, v in req.headers.items()),
        req.body,
    ))


parser = argparse.ArgumentParser(description='A tool to determine the health of a node in Consul and report to Nagios')
parser.add_argument('hostname', type=str, help='The Hostname')

hostname = vars(parser.parse_args())['hostname']

if hostname.find("justin.tv") != -1:
    hostname = re.sub('\.justin.tv$', '', hostname)
else:
    print "UNKNOWN: Invalid Hostname"
    sys.exit(3)

if '.' not in hostname:
    print "UNKNOWN: Invalid Hostname"
    sys.exit(3)
hostname = hostname.split('.', 1)
r = do_request('/health/node/',hostname[0],hostname[1])['response']
try:
    node_status = json.loads(r.text)
except:
    print "WARN: Error querying node status from Consul API"
    sys.exit(1)

if len(node_status) < 1:
    print "WARN: Empty output from node status query to Consul API"
    exit(1)

statuses = list()
errors = list()
for s in node_status:
    if 'serfHealth' in s['CheckID']:
      if '200 OK' in s['Output']:
          statuses.append('health checks okay')
      else:
          statuses.append(s['Output'].strip())
      if s["Status"] != "passing":
          statuses.append("Status=%s" % s["Status"])
          errors.append(s['Output'].strip())


r_query = do_request('/catalog/node/',hostname[0],hostname[1])
# Break the query into the actual response, and the request, for debugging
r = r_query['response']
q = r_query['request']

consul_host_ip = json.loads(r_query['body'])['Node']['Address']

if r.status_code != 200:
    errors.append("Received a {} status code from node catalog request \n".format(r.status_code))
    errors.append("Request: {}\n".format(q))
    errors.append("Response: {} \n".format(r.text))
    errors.append("Headers: {} \n".format(r.headers))
if (len(r.text) <= 4) and (r.status_code == 200):
    errors.append("Received empty output from a stale consul catalog request but a 200 response")

try:
    socket.inet_aton(consul_host_ip)
except socket.error as e:
    errors.append("Invalid IP address: {0}".format(e))

if consul_host_ip == '127.0.0.1':
    errors.append("Invalid IP address: Consul has local address! {0}".format(consul_host_ip))

real_host_ips = os.popen('ip addr | grep "\<inet\>" | awk \'{ print $2 }\' | awk -F "/" \'{ print $1 }\' | grep -v 127.0.0.1').read().strip().split('\n')

if consul_host_ip not in real_host_ips:
    errors.append("Invalid IP address: Consul IP not in list of interfaces! {0}".format(real_host_ips))


if errors:
    errors = " ".join(errors)
    statuses = "/".join(statuses)
    statuses = statuses.replace('\n', ' ')
    print "CRIT: {0} errors:\n {1} Status:\n {2}".format(".".join(hostname), errors, statuses)
    sys.exit(2)
elif statuses:
    statuses = "/".join(statuses)
    statuses = statuses.replace('\n', ' ')
    print "OK: {0} status: {1}".format(".".join(hostname), statuses)
    sys.exit(0)
else:
    print "WARN: No Check Found"
    sys.exit(1)
