#!/usr/bin/env python

import requests
import sys
import argparse
import re

R_OK = 0
R_ERROR = 2
R_UNKNOWN = 3
# regexp for the valid single fqdn label
LDH_RE = re.compile('^[a-z0-9]([a-z0-9-]{0,61}[a-z0-9])?$', re.IGNORECASE)


class NoLeaderException(Exception):
    pass


class CantTalkToDcException(Exception):
    pass


class BadNodeNameException(Exception):
    pass


#this function checks if this node can find a leader.
def check_leader_exists(port):
    URL = 'http://127.0.0.1:%d/v1/status/leader' % port
    r = requests.get(URL)
    data = r.json
    if data is None:
        print 'no leader found'
        raise NoLeaderException()
    print 'current leader is %s' % data


def check_current_dc(port):
    URL = 'http://127.0.0.1:%d/v1/agent/self' % port
    data = requests.get(URL).json
    dc = data['Member']['Tags']['dc']
    return dc


def is_valid_dc_json(json):
    return len(json) >= 2

def test_dc(port):
    dc = check_current_dc(port)
    URL = 'http://127.0.0.1:%d/v1/catalog/service/consul?dc=%s' % (port,dc)
    data = requests.get(URL).json
    if not is_valid_dc_json(data):
        print data
        raise CantTalkToDcException()

def validate_fqdn(dn):
    if dn.endswith('.'):
        dn = dn[:-1]
    if len(dn) < 1 or len(dn) > 253:
        return False
    return all(LDH_RE.match(x) for x in dn.split('.'))

def check_node_names(port):
    URL = 'http://127.0.0.1:%d/v1/catalog/nodes' % port
    data = requests.get(URL).json
    for node in data:
        if not validate_fqdn(node['Node']):
            print('Found node with invalid name: "{}"'.format(node['Node']))
            raise BadNodeNameException()

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-p', '--port', type=int, default=8500, help='port consul is running on')
    args = parser.parse_args()
    port = args.port
    try:
        check_leader_exists(port)
        check_node_names(port)
        test_dc(port)
        return R_OK
    except (NoLeaderException, CantTalkToDcException, BadNodeNameException):
        return R_ERROR
    except Exception as e:
        print(e)
        return R_UNKNOWN

if __name__ == "__main__":
    sys.exit(main())
