#!/usr/bin/env python

# Adapted from https://github.com/etsy/nagios_tools/

import argparse
import getopt
import nagiosplugin
import sys
import requests

DEFAULT_SECONDS = 60 * 5


class GraphiteMetric(nagiosplugin.Resource):
    def __init__(self, url, seconds):
        self.url = url
        self.seconds = seconds

    def probe(self):
        val = self.get_value(self.url, self.seconds)
        return [nagiosplugin.Metric('avg', val, context="graphite")]

    def get_value(self, url, seconds=0):
        """Get the value from a Graphite graph"""

        try:
            resp = requests.get(url, timeout=15)
            assert resp.status_code == 200, "{} opening {}".format(resp.code, resp.url)

        except IOError, e:
            # socket error frm urlopen
            print e
            sys.exit(3)

        except AssertionError, e:
            # urlopen returned http error code in e
            print 'ERROR: HTTP error:', e
            sys.exit(3)

        data = resp.text

        data_value = self.eval_graphite_data(data, seconds)
        return data_value

    def eval_graphite_data(self, data, seconds):
        """Get the most recent correct value from the data"""

        try:
            sample_period = int(data.split('|')[0].split(',')[-1])
            all_data_points = data.split('|')[-1].split(',')
        except ValueError, e:
            # eval_graphite_data barfed parsing output
            print 'ERROR: Parse error on graphite output:', e
            sys.exit(3)


        # Evaluate what graphite returned, should either be a float, or None
        # First, if the number of seconds of data we want to examine is smaller
        # or equals the graphite sample period, just grab the latest data
        # point.
        # If that data point is None, grab the one before it.
        # If that is None too, return 0.0.
        if seconds <= sample_period:
            if eval(all_data_points[-1]):
                data_value = float(all_data_points[-1])
            elif eval(all_data_points[-2]):
                data_value = float(all_data_points[-2])
            else:
                data_value = 0.0
        else:
            # Second, if we requested more than on graphite sample period, work
            # out how many sample periods we wanted (python always rounds
            # division *down*)
            data_points = (seconds/sample_period)
            data_set = [
                float(x) for x in all_data_points[-data_points:] if eval(x)
            ]
            if data_set:
                data_value = float(sum(data_set) / len(data_set))
            else:
                data_value = 0.0
        return data_value


def graphite_url(host, metric):
    if host.find('justin.tv') == -1 and host[-1] != '/':
        host = host + ".justin.tv"

    if host.find('http') == -1:
        host = "http://" + host

    return "%s/render?target=%s&rawData" % (host, metric)


def parse_args():
    argp = argparse.ArgumentParser(add_help=False)
    argp.add_argument('-h', '--host', required=True)
    argp.add_argument('-m', '--metric', required=True)
    argp.add_argument('-w', '--warn', metavar='WARN RANGE')
    argp.add_argument('-c', '--crit', metavar='CRITICAL RANGE')
    argp.add_argument('-s', '--seconds', type=int, default=DEFAULT_SECONDS)
    return argp.parse_args()


@nagiosplugin.guarded
def main():
    args = parse_args()
    url = graphite_url(args.host, args.metric)
    check = nagiosplugin.Check(
        GraphiteMetric(url, args.seconds),
        nagiosplugin.ScalarContext("graphite", args.warn, args.crit))
    check.main()

if __name__ == '__main__':
    main()

