#!/usr/bin/env python

import time
import csv
import re
import requests as r
import json
import argparse


class CheckHAProxyBackends():
    """Checks HAProxy Backends"""
    parser = argparse.ArgumentParser()
    parser.add_argument("--requests-timeout", help="User configured HTTP timeout",
        default=30, type=int)
    parser.add_argument("--ignore-backends", help="A comma separated list of backends to ignore",
        default='birdbot,rabbit-cluster-site-staging,rabbit-cluster-site-staging-http-api,\
        rails-session-memcached-shard,static-jtvfs,stats-master,version,sitedb,\
        vod')
    parser.add_argument("--haproxy-backend-url", help="The HAProxy Base URL for Stats",
        default='http://127.0.0.1:2015')
    parser.add_argument("--username", help="The HAProxy Stats Username",
        default='oxygen')
    parser.add_argument("--password", help="The HAProxy Stats Password",
        default='atom')

    options = parser.parse_args()

    def nagios_exit(self, status, message):
        """pynagios is not Python3 compatible, so writing our own"""
        status_dict = { 'OK' : 0, 'WARN': 1, 'CRITICAL': 2, 'UNKNOWN': 3 }

        try:
            if status in status_dict:
                print("{}: {}".format(status, message))
                exit(status_dict[status])
            else:
                print("Err: {} is invalid status, {}".format(status, message))
                exit(status_dict["UNKNOWN"])
        except Exception as err:
                print("Err: parsing status status, {}".format(err))
                exit(status_dict["UNKNOWN"])

    def nagios_error_status(self, args):
        """Function to catch general exceptions not covered by other exceptions"""
        return self.nagios_exit('UNKNOWN', 'Check Script Error: ' + args)

    def check_haproxy_backends(self, haproxy_backend_url, requests_timeout, username, password):
        """Query HAProxy for CSV stats output and return it"""
        return_output = {}
        ctime = int(time.time())
        stats_uri = "/stats;csv"

        s = r.Session()
        s.trust_env = False #Disable System Proxy
        url = "{}{}".format(haproxy_backend_url, stats_uri)
        try:
            return s.get(url, timeout=requests_timeout, auth=(username, password))
        except Exception as err:
            return self.nagios_exit('UNKNOWN', "Error contacting HAProxy: {}".format(err))

    def parse_csv_data(self, csv_data):
        """Parse CSV Dict and Organize the data into UP/Down by Service"""
        iter_item = 0
        services = {}

        for row in csv_data:
            iter_item +=1
            if iter_item > 1: # CSV Header is in the first line. Ignore it.
                service = row[0]
                if service not in services.keys():
                    services[service] = {}
                    services[service]['up'] = 0
                    services[service]['down'] = 0
                if "UP" in row:
                    services[service]['up'] +=1
                if "DOWN" in row:
                    services[service]['down'] += 1
        return services


    def import_csv(self, haproxy_csv_output):
        """ Parse CSV with Python CSV Library"""
        cr = csv.reader(haproxy_csv_output.splitlines(), delimiter=',')
        return list(cr)

    def check(self):
        """Nagios Check to Validate the number of UP/DOWN HAProxy Backends"""
        try:
            error_status = 0
            cluster_state = []

            haproxy_resource = self.check_haproxy_backends(self.options.haproxy_backend_url,
                self.options.requests_timeout, self.options.username, self.options.password)

            if haproxy_resource.status_code != 200:
                raise Exception("Response Code {}".format(haproxy_resource.status_code))

            # Haproxy emits backends data in CSV format
            csv_data = self.import_csv(haproxy_resource.text)
            # Break out the parsed CSV into a dict
            service_status_dict = self.parse_csv_data(csv_data)
            # Splt ignore backends into a list
            ignore_backends = self.options.ignore_backends.split(',')
            # Parse services for UP/DOWN Backends
            for service in service_status_dict:
                if not any([x.strip() in service for x in ignore_backends]):
                    service_total = service_status_dict[service]['down'] + service_status_dict[service]['up']
                    if service_status_dict[service]['down'] > 0:
                        error_status = 1
                        cluster_state.append("CRITICAL {} has {}/{} HAProxy Backends".format(
                            service, service_status_dict[service]['up'], service_total))
                    if service_status_dict[service]['down'] == 0:
                        cluster_state.append("OK {} has {}/{} HAProxy Backends".format(
                            service, service_status_dict[service]['up'], service_total))

        except Exception as err:
            return self.nagios_error_status(
                'Unable to determine HAProxy resources: {0}'.format(err))

        cluster_state.sort()
        check_output = """{}""".format("\n".join(cluster_state[0:]))

        if error_status > 0:
            return self.nagios_exit('CRITICAL', "HAProxy Backend Failure(s)\n\n{}".format(check_output))
        else:
            return self.nagios_exit('OK', "All HAProxy Backends are Healthy\n\n{}".format(check_output))


if __name__ == "__main__":
    # Instantiate the plugin, check it, and then exit
    CheckHAProxyBackends().check().exit()
