#!/usr/bin/env python
"""Nagios check for ratio of haproxy current to max connections."""
import argparse
import socket

import nagiosplugin


class HaproxyConnectionRatio(nagiosplugin.Resource):
    """Determines the ratio of current to max connections in haproxy."""

    def __init__(self, socket_path):
        self.socket_path = socket_path

    def show_info(self):
        """Returns the output of 'show info' on an haproxy stats socket in a dict."""
        try:
            client = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
            client.connect(self.socket_path)
            client.sendall('show info\n')
            info = client.recv(1024)

            lines = info.split('\n')
            return {l.split(': ')[0]: l.split(': ')[1] for l in lines if l}
        finally:
            client.close()

    def probe(self):
        """Returns ratio of current to max connections."""
        info = self.show_info()

        cur_conn = int(info['CurrConns'])
        max_conn = int(info['Maxconn'])
        conn_ratio = float(cur_conn) / max_conn

        # Intended to provide useful info in summary, not evaluated.
        yield nagiosplugin.Metric('current', cur_conn, min=0, context='null')
        yield nagiosplugin.Metric('max', max_conn, min=0, context='null')
        # This metric is evaluated.
        yield nagiosplugin.Metric('ratio', conn_ratio, min=0, context='ratio')


class HaproxyConnectionRatioSummary(nagiosplugin.Summary):
    """Status line conveying haproxy connection ratio information."""

    def ok(self, results):
        return "current: %s, max: %s, ratio: %s" % \
               (results['current'].metric, results['max'].metric, results['ratio'].metric)

    def problem(self, results):
        return "current: %s, max: %s, ratio: %s" % \
               (results['current'].metric, results['max'].metric, results['ratio'].metric)


@nagiosplugin.guarded
def main():
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument('-w', '--warning', metavar='RANGE', default='',
                        help='return warning if load is outside RANGE')
    parser.add_argument('-c', '--critical', metavar='RANGE', default='',
                        help='return critical if load is outside RANGE')
    parser.add_argument('-s', '--socket_path', required=True,
                        help='haproxy socket path')
    args = parser.parse_args()

    check = nagiosplugin.Check(
        HaproxyConnectionRatio(args.socket_path),
        nagiosplugin.ScalarContext('ratio', args.warning, args.critical),
        HaproxyConnectionRatioSummary())
    check.main()

if __name__ == '__main__':
    main()
