#!/usr/bin/env python

"""
This nagios check should run on each media server and check the following againist the db(slave).

- verify jbods are mounted (critical)
- list any new/unknown jbods (warn)
- verify jbods are mounted on the correct mount point (critical)
- verify store symlink exists and link to the correct jbod (critical)
- display if a jbod/store is enabled/disabled

https://twitchtv.atlassian.net/browse/SYS-655
"""

import sys
sys.path.append('/home/jtv/pylibs/')
import os
from socket import gethostname
import os.path
from optparse import OptionParser
import subprocess

from ttv.nagios import nagios_note, NOK, NWARNING, NCRITICAL
from ttv.db import Table
from ttv.sitedb import SiteDbRo
from ttv.config import add_verbosity_options, set_log_level


JBOD_AVAILABLE_DIR = '/mnt/media/arrays-available'
JBOD_ENABLED_DIR = '/mnt/media/arrays-enabled'


def is_uuid_mounted(uuid):
    """return True if uuid is mounted, False if not"""
    return subprocess.call(['findfs', 'UUID=%s' % uuid], stdout=open('/dev/null', 'w')) == 0


def check_jbods():
    """verify that everything is in order"""
    
    # get media_server_id
    this_host = gethostname().split('.')[0]
    media_servers = dict(Table(SiteDbRo(), 'media_servers', ('name', 'id')).get_tuples())
    try:
        this_media_server_id = media_servers[this_host]
    except KeyError:
        nagios_note(NWARNING, 'host "%s" not found in media_servers table!' % this_host)
        sys.exit()
    
    # check that all expected jbod mountpoints are indeed mountpoints
    # (look only at jbods relevant to this_media_server_id)
    jbods = dict([(uuid, mount)
                  for uuid, mount, media_server_id in Table(SiteDbRo(), 'jbods', 
                                                            ('uuid', 'mount', 'media_server_id')).get_tuples() 
                  if media_server_id == this_media_server_id])
    
    for uuid, mount in jbods.items():
        mountpoint = os.path.join(JBOD_AVAILABLE_DIR, uuid)
        if not os.path.ismount(mountpoint):
            nagios_note(NCRITICAL, '%s is supposed to have a jbod mounted on it, but does not' % mountpoint)

        if not is_uuid_mounted(uuid):
            nagios_note(NCRITICAL, '%s is not mounted' % uuid)

    # check for undocumented jbods
    dirnames = os.walk(JBOD_AVAILABLE_DIR).next()[1] + os.walk(JBOD_AVAILABLE_DIR).next()[2]
    for dirname in dirnames:
        if dirname not in jbods.keys():
            nagios_note(NWARNING, 'what is %s doing in %s?' % (dirname, JBOD_AVAILABLE_DIR))

    # check that desired symlinks are present and point to the right filesystem
    for uuid, mount in jbods.items():
        link_name = os.path.join(JBOD_ENABLED_DIR, mount) 
        mount_name = os.path.join(JBOD_AVAILABLE_DIR, uuid)
        if not os.path.islink(link_name) \
                or os.stat(link_name).st_ino != os.stat(mount_name).st_ino:
            nagios_note(NCRITICAL, '%s should be symlink to %s but is not' % link_name)
            
    # check that there's no extraneous junk in the ENABLED dir
    linknames = os.walk(JBOD_ENABLED_DIR).next()[1] + os.walk(JBOD_ENABLED_DIR).next()[2]
    for linkname in linknames:
        if linkname not in jbods.values() and linkname != 'default' and os.path.islink(linkname):
            nagios_note(NWARNING, 'what is %s doing in %s?' % (linkname, JBOD_ENABLED_DIR))
    
    # display which jbods are enabled and disabled
    dirnames = os.walk(JBOD_ENABLED_DIR).next()[1]
    enabled = [d for d in dirnames if d.startswith('store')]
    nagios_note(NOK, 'enabled: %s' % enabled)


def parse_args(argv):
    """parse command line"""

    parser = OptionParser()

    add_verbosity_options(parser)

    (options, args) = parser.parse_args(argv)

    # verbosity
    set_log_level(options.verbose - options.quiet)

    # check args
    if len(args) > 1:
        parser.error('extraneous arguments: %s' % args[1:])

    return (options, args)


def main(argv):
    """main"""
    options, args = parse_args(argv)

    check_jbods()
    return 0


if __name__ == '__main__':
    sys.exit(main(sys.argv))





