#!/usr/bin/env python2.7
""" Check how many users are connected to the OpenVPN AS. """
# -*- coding: utf-8 -*-

import os
import locale
import subprocess as sp
import argparse
import ast

locale.setlocale(locale.LC_ALL, 'en_US')


def liverun(cmd):
    """ Create a generator that returns a tuple of stdout and stderr. """
    env = dict(os.environ)

    # Combining stdout and stderr. I can't find a way to keep both separate
    # while getting the data 'live'. itertools.izip_longest seemed like it'd
    # almost do it but it caches the results before sending it out...
    proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.STDOUT, env=env)

    return " ".join(proc.stdout.readlines())


def main():
    """ Run the check. """
    parser = argparse.ArgumentParser(description="Check OpenVPN AS Connected Users")
    parser.add_argument('-c', '--crit', type=str, default="95%",
                        help='Crit usage level as percentage, or as absolute value')
    parser.add_argument('-w', '--warn', type=str, default="85%",
                        help='Warn usage level as percentage, or as absolute value')
    args = parser.parse_args()

    try:
        result = liverun(["/usr/bin/sudo", "/usr/local/openvpn_as/scripts/sacli", "LicUsage"])
        used, avail = ast.literal_eval(result)

    except SyntaxError:
        print "UNKNOWN: Error parsing sacli LicUsage output."
        print "Try this: sudo /usr/local/openvpn_as/scripts/sacli LicUsage"
        exit(3)

    # Accept a percentage of the total or an absolute value.
    if "%" in args.warn:
        warn_num = int(float(args.warn.strip("%")) / 100 * avail)
    else:
        warn_num = int(args.warn)

    if "%" in args.crit:
        crit_num = int(float(args.crit.strip("%")) / 100 * avail)
    else:
        crit_num = int(args.crit)

    pct_used = float(used) / avail

    msg = "OK"
    # Check values
    if used > crit_num:
        msg = "CRITICAL"
    elif used > warn_num:
        msg = "WARNING"

    # Return result
    print "{msg}: {used} of {avail} connected ({pct:0.1%} utilized)".format(
        msg=msg, used=used, avail=avail, pct=pct_used)
    # Nagios perf data.
    print "| users_connected={used};{warn_num};{crit_num};{min};{max}".format(
        used=used, warn_num=warn_num, crit_num=crit_num, min=0, max=avail)

    if msg == "CRITICAL":
        exit(2)
    if msg == "WARNING":
        exit(1)


main()
